@extends('layouts.app')

@section('title', 'Detalhes da Barragem - ' . $barragem->nome)

@section('content')
<div class="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100 py-8">
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">

        <!-- Header Hero Section -->
        <div class="relative bg-white rounded-3xl shadow-xl overflow-hidden mb-8">
            <div class="absolute inset-0 bg-gradient-to-r from-blue-600 via-blue-700 to-indigo-800"></div>
            <div class="absolute inset-0 bg-black/20"></div>

            <!-- Decorative background pattern -->
            <div class="absolute inset-0 opacity-10">
                <svg class="w-full h-full" viewBox="0 0 100 100" xmlns="http://www.w3.org/2000/svg">
                    <defs>
                        <pattern id="grid" width="10" height="10" patternUnits="userSpaceOnUse">
                            <path d="M 10 0 L 0 0 0 10" fill="none" stroke="white" stroke-width="1"/>
                        </pattern>
                    </defs>
                    <rect width="100" height="100" fill="url(#grid)" />
                </svg>
            </div>

            <div class="relative px-8 py-12">
                <!-- Breadcrumb -->
                <nav class="flex mb-6" aria-label="Breadcrumb">
                    <ol class="inline-flex items-center space-x-2">
                        <li><a href="{{ route('dashboard') }}" class="text-blue-200 hover:text-white transition-colors"><i class="fas fa-home mr-1"></i>Dashboard</a></li>
                        <li><i class="fas fa-chevron-right text-blue-300 mx-4"></i></li>
                        <li><a href="{{ route('barragens.index') }}" class="text-blue-200 hover:text-white transition-colors">Barragens</a></li>
                        <li><i class="fas fa-chevron-right text-blue-300 mx-4"></i></li>
                        <li><span class="text-white font-semibold">{{ $barragem->nome }}</span></li>
                    </ol>
                </nav>

                <div class="grid grid-cols-1 lg:grid-cols-3 gap-8 items-start">
                    <!-- Main Info -->
                    <div class="lg:col-span-2">
                        <div class="flex items-start space-x-6">
                            <div class="flex-shrink-0 w-20 h-20 bg-gray-800/80 backdrop-blur rounded-2xl flex items-center justify-center">
                                <i class="fas fa-dam text-white text-3xl"></i>
                            </div>
                            <div class="flex-1">
                                <h1 class="text-4xl font-black text-white mb-2">{{ $barragem->nome }}</h1>
                                <p class="text-blue-200 text-lg mb-4">Código: <span class="font-mono font-bold">{{ $barragem->codigo }}</span></p>

                                <!-- Status badges -->
                                <div class="flex flex-wrap gap-3 mb-6">
                                    @php
                                        $estadoConfig = [
                                            'activa' => ['class' => 'bg-green-500/90 text-white border-green-400', 'icon' => 'fas fa-check-circle'],
                                            'inactiva' => ['class' => 'bg-red-500/90 text-white border-red-400', 'icon' => 'fas fa-times-circle'],
                                            'manutencao' => ['class' => 'bg-yellow-500/90 text-white border-yellow-400', 'icon' => 'fas fa-wrench']
                                        ];
                                        $config = $estadoConfig[$barragem->estado] ?? ['class' => 'bg-gray-500/90 text-white border-gray-400', 'icon' => 'fas fa-question-circle'];
                                    @endphp

                                    <span class="inline-flex items-center px-4 py-2 rounded-full text-sm font-bold backdrop-blur border {{ $config['class'] }}">
                                        <i class="{{ $config['icon'] }} mr-2"></i>
                                        {{ ucfirst($barragem->estado) }}
                                    </span>

                                    @if($barragem->bacia_hidrografica)
                                    <span class="inline-flex items-center px-4 py-2 rounded-full text-sm font-bold bg-cyan-500/90 text-white backdrop-blur border border-cyan-400">
                                        <i class="fas fa-water mr-2"></i>
                                        {{ $barragem->bacia_hidrografica->nome }}
                                    </span>
                                    @endif

                                    @if($barragem->ano_construcao)
                                    <span class="inline-flex items-center px-4 py-2 rounded-full text-sm font-bold bg-slate-600/90 text-white backdrop-blur border border-slate-400">
                                        <i class="fas fa-calendar mr-2"></i>
                                        {{ $barragem->ano_construcao }}
                                    </span>
                                    @endif
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Quick Actions -->
                    <div class="lg:col-span-1">
                        <div class="bg-gray-800/80 backdrop-blur-md rounded-2xl p-6 border border-gray-600/40">
                            <h3 class="text-white font-bold text-lg mb-4">Ações Rápidas</h3>
                            <div class="grid grid-cols-2 lg:grid-cols-1 gap-3">
                                <a href="{{ route('barragens.leituras', $barragem) }}"
                                   class="flex items-center justify-center px-4 py-3 bg-gray-700/90 hover:bg-gray-600/90 text-white rounded-xl transition-all backdrop-blur font-medium">
                                    <i class="fas fa-chart-line mr-2"></i>
                                    Leituras
                                </a>
                                <a href="{{ route('barragens.edit', $barragem) }}"
                                   class="flex items-center justify-center px-4 py-3 bg-gray-700/90 hover:bg-gray-600/90 text-white rounded-xl transition-all backdrop-blur font-medium">
                                    <i class="fas fa-edit mr-2"></i>
                                    Editar
                                </a>
                                <button onclick="generateReport()"
                                        class="flex items-center justify-center px-4 py-3 bg-gray-700/90 hover:bg-gray-600/90 text-white rounded-xl transition-all backdrop-blur font-medium">
                                    <i class="fas fa-file-alt mr-2"></i>
                                    Relatório
                                </button>
                                <a href="/graficos/barragens"
                                   class="flex items-center justify-center px-4 py-3 bg-gray-700/90 hover:bg-gray-600/90 text-white rounded-xl transition-all backdrop-blur font-medium">
                                    <i class="fas fa-chart-area mr-2"></i>
                                    Gráficos
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        @if($barragem->ultima_leitura)
        <!-- Main Status Dashboard -->
        <div class="bg-white rounded-3xl shadow-xl overflow-hidden mb-8">
            <div class="bg-gradient-to-r from-slate-800 to-slate-900 px-8 py-6">
                <div class="flex items-center justify-between">
                    <div>
                        <h2 class="text-2xl font-black text-white mb-1">Status Operacional</h2>
                        <p class="text-slate-300">Monitoramento em tempo real</p>
                    </div>
                    <div class="bg-gray-800/80 backdrop-blur rounded-xl px-4 py-2">
                        <div class="text-white text-sm font-medium">Última atualização</div>
                        <div class="text-slate-300 text-xs">{{ $barragem->ultima_leitura->data_leitura->format('d/m/Y H:i') }}</div>
                    </div>
                </div>
            </div>

            <!-- Main Metrics Grid -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4">
                <!-- Cota Atual -->
                <div class="bg-gradient-to-br from-blue-50 to-blue-100 p-8 border-r border-gray-200">
                    <div class="flex items-center justify-between mb-6">
                        <div class="w-16 h-16 bg-blue-600 rounded-2xl flex items-center justify-center shadow-lg">
                            <i class="fas fa-ruler-vertical text-white text-2xl"></i>
                        </div>
                        <div class="text-right">
                            <div class="text-xs font-bold text-blue-700 uppercase tracking-wider mb-1">Cota Atual</div>
                            <div class="text-4xl font-black text-blue-900 leading-none">{{ number_format($barragem->ultima_leitura->cota_actual, 1) }}</div>
                            <div class="text-sm font-bold text-blue-600 mt-1">metros</div>
                        </div>
                    </div>
                    @if($barragem->cota_maxima)
                        @php $cotaPerc = ($barragem->ultima_leitura->cota_actual / $barragem->cota_maxima) * 100; @endphp
                        <div class="space-y-2">
                            <div class="flex justify-between text-xs font-semibold text-blue-700">
                                <span>Máx: {{ number_format($barragem->cota_maxima, 1) }}m</span>
                                <span>{{ number_format($cotaPerc, 1) }}%</span>
                            </div>
                            <div class="w-full bg-blue-200/60 rounded-full h-3 overflow-hidden">
                                <div class="bg-gradient-to-r from-blue-600 to-blue-800 h-full rounded-full transition-all duration-1000 ease-out"
                                     style="width: {{ min(100, $cotaPerc) }}%"></div>
                            </div>
                        </div>
                    @endif
                </div>

                <!-- Volume Atual -->
                <div class="bg-gradient-to-br from-emerald-50 to-emerald-100 p-8 border-r border-gray-200">
                    <div class="flex items-center justify-between mb-6">
                        <div class="w-16 h-16 bg-emerald-600 rounded-2xl flex items-center justify-center shadow-lg">
                            <i class="fas fa-tint text-white text-2xl"></i>
                        </div>
                        <div class="text-right">
                            <div class="text-xs font-bold text-emerald-700 uppercase tracking-wider mb-1">Volume Atual</div>
                            <div class="text-4xl font-black text-emerald-900 leading-none">
                                {{ $barragem->ultima_leitura->volume_actual ? number_format($barragem->ultima_leitura->volume_actual, 1) : '--' }}
                            </div>
                            <div class="text-sm font-bold text-emerald-600 mt-1">Mm³</div>
                        </div>
                    </div>
                    @if($barragem->volume_maximo && $barragem->ultima_leitura->volume_actual)
                        @php $volPerc = ($barragem->ultima_leitura->volume_actual / $barragem->volume_maximo) * 100; @endphp
                        <div class="space-y-2">
                            <div class="flex justify-between text-xs font-semibold text-emerald-700">
                                <span>Máx: {{ number_format($barragem->volume_maximo, 1) }} Mm³</span>
                                <span>{{ number_format($volPerc, 1) }}%</span>
                            </div>
                            <div class="w-full bg-emerald-200/60 rounded-full h-3 overflow-hidden">
                                <div class="bg-gradient-to-r from-emerald-600 to-emerald-800 h-full rounded-full transition-all duration-1000 ease-out"
                                     style="width: {{ min(100, $volPerc) }}%"></div>
                            </div>
                        </div>
                    @endif
                </div>

                <!-- Enchimento -->
                <div class="bg-gradient-to-br from-purple-50 to-purple-100 p-8 border-r border-gray-200">
                    @php
                        $percentage = $barragem->ultima_leitura->percentagem_enchimento ?? 0;
                        $statusColor = $percentage >= 90 ? 'red' : ($percentage >= 70 ? 'yellow' : ($percentage >= 50 ? 'green' : 'blue'));
                        $statusText = $percentage >= 90 ? 'Crítico' : ($percentage >= 70 ? 'Alto' : ($percentage >= 50 ? 'Normal' : 'Baixo'));
                    @endphp
                    <div class="flex items-center justify-between mb-6">
                        <div class="w-16 h-16 bg-purple-600 rounded-2xl flex items-center justify-center shadow-lg">
                            <i class="fas fa-percentage text-white text-2xl"></i>
                        </div>
                        <div class="text-right">
                            <div class="text-xs font-bold text-purple-700 uppercase tracking-wider mb-1">Enchimento</div>
                            <div class="text-4xl font-black text-purple-900 leading-none">{{ number_format($percentage, 1) }}</div>
                            <div class="text-sm font-bold text-purple-600 mt-1">por cento</div>
                        </div>
                    </div>
                    <div class="space-y-2">
                        <div class="flex justify-between text-xs font-semibold text-purple-700">
                            <span>Status: {{ $statusText }}</span>
                            <span>{{ number_format($percentage, 1) }}%</span>
                        </div>
                        <div class="w-full bg-purple-200/60 rounded-full h-3 overflow-hidden">
                            <div class="bg-gradient-to-r from-purple-600 to-purple-800 h-full rounded-full transition-all duration-1000 ease-out"
                                 style="width: {{ min(100, $percentage) }}%"></div>
                        </div>
                    </div>
                </div>

                <!-- Status Leitura -->
                <div class="bg-gradient-to-br from-slate-50 to-slate-100 p-8">
                    @php $diasAtras = $barragem->ultima_leitura->data_leitura->diffInDays(now()); @endphp
                    <div class="flex items-center justify-between mb-6">
                        <div class="w-16 h-16 bg-slate-600 rounded-2xl flex items-center justify-center shadow-lg">
                            <i class="fas fa-satellite text-white text-2xl"></i>
                        </div>
                        <div class="text-right">
                            <div class="text-xs font-bold text-slate-700 uppercase tracking-wider mb-1">Última Leitura</div>
                            <div class="text-2xl font-black text-slate-900 leading-none">{{ $barragem->ultima_leitura->data_leitura->format('d/m') }}</div>
                            <div class="text-sm font-bold text-slate-600 mt-1">{{ $barragem->ultima_leitura->hora_leitura ?? '06:00' }}</div>
                        </div>
                    </div>
                    <div class="space-y-2">
                        <div class="flex justify-between text-xs font-semibold text-slate-700">
                            <span>{{ ucfirst($barragem->ultima_leitura->metodo_leitura ?? 'Manual') }}</span>
                            @if($diasAtras == 0)
                                <span class="px-2 py-1 bg-green-100 text-green-800 rounded-full font-bold">Hoje</span>
                            @elseif($diasAtras <= 1)
                                <span class="px-2 py-1 bg-yellow-100 text-yellow-800 rounded-full font-bold">{{ $diasAtras }}d</span>
                            @else
                                <span class="px-2 py-1 bg-red-100 text-red-800 rounded-full font-bold">{{ $diasAtras }}d</span>
                            @endif
                        </div>
                        <div class="w-full bg-slate-200/60 rounded-full h-3 overflow-hidden">
                            @if($diasAtras == 0)
                                <div class="bg-gradient-to-r from-green-500 to-green-600 h-full rounded-full w-full transition-all duration-1000"></div>
                            @elseif($diasAtras <= 3)
                                <div class="bg-gradient-to-r from-yellow-500 to-yellow-600 h-full rounded-full w-3/4 transition-all duration-1000"></div>
                            @else
                                <div class="bg-gradient-to-r from-red-500 to-red-600 h-full rounded-full w-1/2 transition-all duration-1000"></div>
                            @endif
                        </div>
                    </div>
                </div>
            </div>
        </div>
        @endif

        <!-- Content Grid -->
        <div class="grid grid-cols-1 xl:grid-cols-4 gap-8">

            <!-- Main Content Area -->
            <div class="xl:col-span-3 space-y-8">

                <!-- Technical Specifications Cards Grid -->
                <div class="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-3 gap-6">

                    <!-- Cotas e Níveis -->
                    <div class="bg-white rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden">
                        <div class="bg-gradient-to-br from-blue-600 to-indigo-700 px-6 py-4">
                            <div class="flex items-center">
                                <div class="w-10 h-10 bg-white/20 rounded-xl flex items-center justify-center mr-3">
                                    <i class="fas fa-ruler-vertical text-white"></i>
                                </div>
                                <h3 class="text-xl font-black text-white">Cotas & Níveis</h3>
                            </div>
                        </div>
                        <div class="p-6 space-y-3">
                            <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium">Máxima</span>
                                <span class="font-black text-lg text-gray-900">{{ $barragem->cota_maxima ? number_format($barragem->cota_maxima, 1) . 'm' : 'N/A' }}</span>
                            </div>
                            <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium">Mínima</span>
                                <span class="font-black text-lg text-gray-900">{{ $barragem->cota_minima ? number_format($barragem->cota_minima, 1) . 'm' : 'N/A' }}</span>
                            </div>
                            @if($barragem->cota_npa)
                            <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium">NPA</span>
                                <span class="font-black text-lg text-gray-900">{{ number_format($barragem->cota_npa, 1) }}m</span>
                            </div>
                            @endif
                            @if($barragem->cota_nme)
                            <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium">NME</span>
                                <span class="font-black text-lg text-gray-900">{{ number_format($barragem->cota_nme, 1) }}m</span>
                            </div>
                            @endif
                            @if($barragem->cota_nmc)
                            <div class="flex justify-between items-center py-2">
                                <span class="text-gray-600 font-medium">NMC</span>
                                <span class="font-black text-lg text-gray-900">{{ number_format($barragem->cota_nmc, 1) }}m</span>
                            </div>
                            @endif
                        </div>
                    </div>

                    <!-- Volumes -->
                    <div class="bg-white rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden">
                        <div class="bg-gradient-to-br from-emerald-600 to-green-700 px-6 py-4">
                            <div class="flex items-center">
                                <div class="w-10 h-10 bg-white/20 rounded-xl flex items-center justify-center mr-3">
                                    <i class="fas fa-tint text-white"></i>
                                </div>
                                <h3 class="text-xl font-black text-white">Volumes</h3>
                            </div>
                        </div>
                        <div class="p-6 space-y-3">
                            <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium">Máximo</span>
                                <div class="text-right">
                                    <div class="font-black text-lg text-gray-900">{{ $barragem->volume_maximo ? number_format($barragem->volume_maximo, 1) : 'N/A' }}</div>
                                    <div class="text-xs text-gray-500">Mm³</div>
                                </div>
                            </div>
                            @if($barragem->volume_util)
                            <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium">Útil</span>
                                <div class="text-right">
                                    <div class="font-black text-lg text-gray-900">{{ number_format($barragem->volume_util, 1) }}</div>
                                    <div class="text-xs text-gray-500">Mm³</div>
                                </div>
                            </div>
                            @endif
                            <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium">Morto</span>
                                <div class="text-right">
                                    <div class="font-black text-lg text-gray-900">{{ $barragem->volume_morto ? number_format($barragem->volume_morto, 1) : 'N/A' }}</div>
                                    <div class="text-xs text-gray-500">Mm³</div>
                                </div>
                            </div>
                            <div class="flex justify-between items-center py-2">
                                <span class="text-gray-600 font-medium">Área Bacia</span>
                                <div class="text-right">
                                    <div class="font-black text-lg text-gray-900">{{ $barragem->area_bacia_hidraulica ? number_format($barragem->area_bacia_hidraulica, 1) : 'N/A' }}</div>
                                    <div class="text-xs text-gray-500">km²</div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Características Físicas -->
                    <div class="bg-white rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden">
                        <div class="bg-gradient-to-br from-purple-600 to-violet-700 px-6 py-4">
                            <div class="flex items-center">
                                <div class="w-10 h-10 bg-white/20 rounded-xl flex items-center justify-center mr-3">
                                    <i class="fas fa-mountain text-white"></i>
                                </div>
                                <h3 class="text-xl font-black text-white">Físicas</h3>
                            </div>
                        </div>
                        <div class="p-6 space-y-3">
                            @if($barragem->altura_barragem)
                            <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium">Altura</span>
                                <div class="text-right">
                                    <div class="font-black text-lg text-gray-900">{{ number_format($barragem->altura_barragem, 1) }}</div>
                                    <div class="text-xs text-gray-500">metros</div>
                                </div>
                            </div>
                            @endif
                            @if($barragem->comprimento_crista)
                            <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium">Comprimento</span>
                                <div class="text-right">
                                    <div class="font-black text-lg text-gray-900">{{ number_format($barragem->comprimento_crista, 0) }}</div>
                                    <div class="text-xs text-gray-500">metros</div>
                                </div>
                            </div>
                            @endif
                            @if($barragem->area_inundacao)
                            <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium">Área Inundação</span>
                                <div class="text-right">
                                    <div class="font-black text-lg text-gray-900">{{ number_format($barragem->area_inundacao/1000000, 1) }}</div>
                                    <div class="text-xs text-gray-500">km²</div>
                                </div>
                            </div>
                            @endif
                            @if($barragem->tipo_barragem)
                            <div class="flex justify-between items-center py-2">
                                <span class="text-gray-600 font-medium">Tipo</span>
                                <span class="font-black text-sm text-gray-900 capitalize">{{ $barragem->tipo_barragem }}</span>
                            </div>
                            @endif
                        </div>
                    </div>

                    <!-- Segurança -->
                    <div class="bg-white rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden">
                        <div class="bg-gradient-to-br from-amber-600 to-orange-700 px-6 py-4">
                            <div class="flex items-center">
                                <div class="w-10 h-10 bg-white/20 rounded-xl flex items-center justify-center mr-3">
                                    <i class="fas fa-shield-alt text-white"></i>
                                </div>
                                <h3 class="text-xl font-black text-white">Segurança</h3>
                            </div>
                        </div>
                        <div class="p-6 space-y-3">
                            @if($barragem->cota_seguranca)
                            <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium">Cota Segurança</span>
                                <span class="font-black text-lg text-gray-900">{{ number_format($barragem->cota_seguranca, 1) }}m</span>
                            </div>
                            @endif
                            <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium">Alerta</span>
                                <span class="font-black text-lg text-gray-900">{{ $barragem->nivel_alerta ? number_format($barragem->nivel_alerta, 1) . 'm' : 'N/A' }}</span>
                            </div>
                            <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium">Emergência</span>
                                <span class="font-black text-lg text-gray-900">{{ $barragem->nivel_emergencia ? number_format($barragem->nivel_emergencia, 1) . 'm' : 'N/A' }}</span>
                            </div>
                            @if($barragem->classe_risco)
                            <div class="flex justify-between items-center py-2">
                                <span class="text-gray-600 font-medium">Classe Risco</span>
                                <span class="inline-flex items-center px-3 py-1 rounded-full text-xs font-bold
                                    @if($barragem->classe_risco == 'alto') bg-red-100 text-red-800
                                    @elseif($barragem->classe_risco == 'medio') bg-yellow-100 text-yellow-800
                                    @else bg-green-100 text-green-800 @endif">
                                    {{ ucfirst($barragem->classe_risco) }}
                                </span>
                            </div>
                            @endif
                        </div>
                    </div>

                    <!-- Localização -->
                    <div class="bg-white rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden">
                        <div class="bg-gradient-to-br from-rose-600 to-pink-700 px-6 py-4">
                            <div class="flex items-center">
                                <div class="w-10 h-10 bg-white/20 rounded-xl flex items-center justify-center mr-3">
                                    <i class="fas fa-map-marker-alt text-white"></i>
                                </div>
                                <h3 class="text-xl font-black text-white">Localização</h3>
                            </div>
                        </div>
                        <div class="p-6 space-y-3">
                            <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium">Província</span>
                                <span class="font-black text-lg text-gray-900">{{ $barragem->provincia ?? 'N/A' }}</span>
                            </div>
                            <div class="flex justify-between items-center py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium">Distrito</span>
                                <span class="font-black text-lg text-gray-900">{{ $barragem->distrito ?? 'N/A' }}</span>
                            </div>
                            @if($barragem->latitude && $barragem->longitude)
                            <div class="py-2 text-center">
                                <x-coordenadas-clicaveis
                                    :latitude="$barragem->latitude"
                                    :longitude="$barragem->longitude"
                                    :nome="$barragem->nome"
                                    tipo="barragem"
                                    :id="$barragem->id"
                                    size="sm" />
                            </div>
                            @endif
                        </div>
                    </div>

                    <!-- Projeto -->
                    @if($barragem->empresa_projetista || $barragem->empresa_construtora || $barragem->ano_projeto)
                    <div class="bg-white rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 overflow-hidden">
                        <div class="bg-gradient-to-br from-slate-600 to-gray-700 px-6 py-4">
                            <div class="flex items-center">
                                <div class="w-10 h-10 bg-white/20 rounded-xl flex items-center justify-center mr-3">
                                    <i class="fas fa-building text-white"></i>
                                </div>
                                <h3 class="text-xl font-black text-white">Projeto</h3>
                            </div>
                        </div>
                        <div class="p-6 space-y-3">
                            @if($barragem->empresa_projetista)
                            <div class="py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium block text-sm">Projetista</span>
                                <span class="font-black text-gray-900">{{ $barragem->empresa_projetista }}</span>
                            </div>
                            @endif
                            @if($barragem->empresa_construtora)
                            <div class="py-2 border-b border-gray-100">
                                <span class="text-gray-600 font-medium block text-sm">Construtora</span>
                                <span class="font-black text-gray-900">{{ $barragem->empresa_construtora }}</span>
                            </div>
                            @endif
                            @if($barragem->ano_projeto)
                            <div class="flex justify-between items-center py-2">
                                <span class="text-gray-600 font-medium">Projeto</span>
                                <span class="font-black text-lg text-gray-900">{{ $barragem->ano_projeto }}</span>
                            </div>
                            @endif
                        </div>
                    </div>
                    @endif
                </div>

                <!-- Chart Section -->
                <div class="bg-white rounded-2xl shadow-xl overflow-hidden">
                    <div class="bg-gradient-to-r from-indigo-600 to-purple-700 px-8 py-6">
                        <div class="flex items-center justify-between">
                            <div>
                                <h3 class="text-2xl font-black text-white mb-1">Análise Temporal</h3>
                                <p class="text-indigo-200">Evolução dos parâmetros principais</p>
                            </div>
                            <div class="flex space-x-2">
                                <button onclick="updateChartPeriod('7d')" class="period-btn px-4 py-2 bg-white/40 text-white rounded-xl text-sm font-bold hover:bg-white/50 transition-all shadow-lg" data-period="7d">7 dias</button>
                                <button onclick="updateChartPeriod('30d')" class="period-btn px-4 py-2 bg-white/20 text-white rounded-xl text-sm font-bold hover:bg-white/30 transition-all shadow-lg" data-period="30d">30 dias</button>
                                <button onclick="updateChartPeriod('1y')" class="period-btn px-4 py-2 bg-white/20 text-white rounded-xl text-sm font-bold hover:bg-white/30 transition-all shadow-lg" data-period="1y">1 ano</button>
                            </div>
                        </div>
                    </div>
                    <div class="p-8">
                        <div class="h-96">
                            <canvas id="damChart" class="w-full h-full"></canvas>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Sidebar -->
            <div class="xl:col-span-1 space-y-6">

                @if($barragem->ultima_leitura)
                <!-- Operational Status -->
                <div class="bg-white rounded-2xl shadow-lg overflow-hidden">
                    <div class="bg-gradient-to-r from-cyan-600 to-blue-700 px-6 py-4">
                        <h3 class="text-lg font-black text-white flex items-center">
                            <i class="fas fa-cog mr-2"></i>
                            Status Operacional
                        </h3>
                    </div>
                    <div class="p-6 space-y-4">
                        @if($barragem->ultima_leitura->caudal_afluente || $barragem->ultima_leitura->caudal_efluente)
                        <div class="bg-cyan-50 rounded-xl p-4">
                            <h4 class="font-bold text-cyan-900 mb-3">Caudais (m³/s)</h4>
                            <div class="space-y-2">
                                @if($barragem->ultima_leitura->caudal_afluente)
                                <div class="flex justify-between">
                                    <span class="text-cyan-700 font-medium">Afluente</span>
                                    <span class="font-black text-cyan-900">{{ number_format($barragem->ultima_leitura->caudal_afluente, 1) }}</span>
                                </div>
                                @endif
                                @if($barragem->ultima_leitura->caudal_efluente)
                                <div class="flex justify-between">
                                    <span class="text-cyan-700 font-medium">Efluente</span>
                                    <span class="font-black text-cyan-900">{{ number_format($barragem->ultima_leitura->caudal_efluente, 1) }}</span>
                                </div>
                                @endif
                            </div>
                        </div>
                        @endif

                        <div class="bg-blue-50 rounded-xl p-4">
                            <h4 class="font-bold text-blue-900 mb-3">Qualidade dos Dados</h4>
                            <div class="space-y-2">
                                <div class="flex justify-between">
                                    <span class="text-blue-700 font-medium">Método</span>
                                    <span class="px-2 py-1 bg-blue-100 text-blue-800 rounded-lg text-xs font-bold">
                                        {{ ucfirst($barragem->ultima_leitura->metodo_leitura ?? 'Manual') }}
                                    </span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                @endif

                <!-- Quick Actions Panel -->
                <div class="bg-white rounded-2xl shadow-lg overflow-hidden">
                    <div class="bg-gradient-to-r from-emerald-600 to-green-700 px-6 py-4">
                        <h3 class="text-lg font-black text-white flex items-center">
                            <i class="fas fa-bolt mr-2"></i>
                            Ações Rápidas
                        </h3>
                    </div>
                    <div class="p-6 space-y-3">
                        <button onclick="addReading()" class="w-full bg-gradient-to-r from-green-500 to-emerald-600 hover:from-green-600 hover:to-emerald-700 text-white rounded-xl py-3 font-bold transition-all shadow-md hover:shadow-lg">
                            <i class="fas fa-plus mr-2"></i>Nova Leitura
                        </button>
                        <a href="{{ route('barragens.leituras', $barragem) }}" class="w-full block bg-gradient-to-r from-blue-500 to-indigo-600 hover:from-blue-600 hover:to-indigo-700 text-white rounded-xl py-3 font-bold transition-all shadow-md hover:shadow-lg text-center">
                            <i class="fas fa-chart-line mr-2"></i>Histórico
                        </a>
                        <button onclick="generateReport()" class="w-full bg-gradient-to-r from-purple-500 to-violet-600 hover:from-purple-600 hover:to-violet-700 text-white rounded-xl py-3 font-bold transition-all shadow-md hover:shadow-lg">
                            <i class="fas fa-file-alt mr-2"></i>Relatório
                        </button>
                    </div>
                </div>

                <!-- System Info -->
                <div class="bg-white rounded-2xl shadow-lg overflow-hidden">
                    <div class="bg-gradient-to-r from-slate-600 to-gray-700 px-6 py-4">
                        <h3 class="text-lg font-black text-white flex items-center">
                            <i class="fas fa-info-circle mr-2"></i>
                            Informações
                        </h3>
                    </div>
                    <div class="p-6 space-y-3">
                        @if($barragem->observacoes)
                        <div class="bg-yellow-50 rounded-xl p-4 border border-yellow-200">
                            <label class="block text-sm font-bold text-yellow-800 mb-2">Observações</label>
                            <div class="text-sm text-yellow-700">{{ $barragem->observacoes }}</div>
                        </div>
                        @endif

                        <div class="space-y-2">
                            <div class="bg-gray-50 rounded-lg p-3">
                                <label class="block text-xs font-bold text-gray-600">Criado</label>
                                <div class="text-sm font-mono">{{ $barragem->created_at->format('d/m/Y') }}</div>
                            </div>
                            <div class="bg-gray-50 rounded-lg p-3">
                                <label class="block text-xs font-bold text-gray-600">Atualizado</label>
                                <div class="text-sm font-mono">{{ $barragem->updated_at->format('d/m/Y') }}</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
let damChart = null;
let currentPeriod = '7d';

function generateReport() {
    alert('Funcionalidade de relatório será implementada');
}

function addReading() {
    window.location.href = '{{ route("barragens.leituras", $barragem) }}';
}

// Chart initialization and functions...
function initializeDamChart() {
    const ctx = document.getElementById('damChart').getContext('2d');

    damChart = new Chart(ctx, {
        type: 'line',
        data: {
            labels: [],
            datasets: []
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'bottom',
                    labels: {
                        usePointStyle: true,
                        padding: 20,
                        font: { weight: 'bold' }
                    }
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.8)',
                    titleFont: { weight: 'bold' },
                    bodyFont: { weight: 'bold' },
                    cornerRadius: 8
                }
            },
            scales: {
                y: {
                    beginAtZero: false,
                    grid: { color: 'rgba(0, 0, 0, 0.05)' },
                    ticks: { font: { weight: 'bold' } }
                },
                x: {
                    grid: { color: 'rgba(0, 0, 0, 0.05)' },
                    ticks: { font: { weight: 'bold' } }
                }
            },
            elements: {
                line: { tension: 0.4 },
                point: { radius: 4, hoverRadius: 8 }
            }
        }
    });

    updateChartData(currentPeriod);
}

function updateChartData(period) {
    const url = `/api/barragens/{{ $barragem->id }}/chart-data?period=${period}`;

    // Show loading state
    const chartContainer = document.getElementById('damChart').parentElement;
    const loadingHTML = chartContainer.innerHTML;
    chartContainer.innerHTML = '<div class="flex items-center justify-center h-96"><div class="animate-spin rounded-full h-12 w-12 border-b-2 border-indigo-600"></div></div>';

    fetch(url)
        .then(response => response.json())
        .then(data => {
            chartContainer.innerHTML = '<canvas id="damChart" class="w-full h-full"></canvas>';
            const ctx = document.getElementById('damChart').getContext('2d');

            // Recreate chart with new data
            if (damChart) {
                damChart.destroy();
            }

            damChart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: data.labels || [],
                    datasets: data.datasets || []
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        title: {
                            display: true,
                            text: `Dados da Barragem - ${period === '7d' ? 'Últimos 7 dias' : period === '30d' ? 'Últimos 30 dias' : 'Último ano'}`,
                            font: { size: 16, weight: 'bold' }
                        },
                        legend: {
                            position: 'bottom',
                            labels: {
                                usePointStyle: true,
                                padding: 20,
                                font: { weight: 'bold' }
                            }
                        },
                        tooltip: {
                            backgroundColor: 'rgba(0, 0, 0, 0.8)',
                            titleFont: { weight: 'bold' },
                            bodyFont: { weight: 'bold' },
                            cornerRadius: 8
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: false,
                            grid: { color: 'rgba(0, 0, 0, 0.05)' },
                            ticks: { font: { weight: 'bold' } },
                            title: {
                                display: true,
                                text: 'Cota (m)',
                                color: 'rgb(59, 130, 246)',
                                font: { weight: 'bold' }
                            }
                        },
                        y1: {
                            type: 'linear',
                            display: true,
                            position: 'right',
                            grid: { drawOnChartArea: false },
                            ticks: { font: { weight: 'bold' } },
                            title: {
                                display: true,
                                text: 'Volume (Mm³)',
                                color: 'rgb(34, 197, 94)',
                                font: { weight: 'bold' }
                            }
                        },
                        y2: {
                            type: 'linear',
                            display: false, // Hidden but used for enchimento
                            min: 0,
                            max: 100
                        },
                        x: {
                            grid: { color: 'rgba(0, 0, 0, 0.05)' },
                            ticks: { font: { weight: 'bold' } }
                        }
                    },
                    elements: {
                        line: { tension: 0.4 },
                        point: { radius: 4, hoverRadius: 8 }
                    }
                }
            });
        })
        .catch(error => {
            console.error('Erro ao carregar dados do gráfico:', error);
            chartContainer.innerHTML = '<div class="flex items-center justify-center h-96 text-red-500"><i class="fas fa-exclamation-triangle mr-2"></i>Erro ao carregar dados do gráfico</div>';
        });
}

function updateChartPeriod(period) {
    currentPeriod = period;

    // Update button styles
    document.querySelectorAll('.period-btn').forEach(btn => {
        btn.classList.remove('bg-white/40');
        btn.classList.add('bg-white/20');
    });

    const selectedBtn = document.querySelector(`[data-period="${period}"]`);
    if (selectedBtn) {
        selectedBtn.classList.remove('bg-white/20');
        selectedBtn.classList.add('bg-white/40');
    }

    updateChartData(period);
}

// Inicialização melhorada
document.addEventListener('DOMContentLoaded', function() {
    // Aguardar um pouco para garantir que todos os elementos estão carregados
    setTimeout(() => {
        const chartCanvas = document.getElementById('damChart');
        if (chartCanvas) {
            initializeDamChart();
        } else {
            console.warn('Canvas do gráfico não encontrado');
        }
    }, 100);
});
</script>
@endsection