@extends('layouts.app')

@section('title', 'Gráficos de Barragens')

@section('content')
<div class="space-y-6">
    <!-- Header -->
    <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between">
            <div>
                <h1 class="text-2xl font-bold text-gray-800 flex items-center">
                    <i class="fas fa-water mr-3 text-blue-600"></i>
                    Gráficos de Barragens
                </h1>
                <p class="text-gray-600 mt-2">Monitoramento de níveis, volumes e enchimento das barragens</p>
            </div>
            <div class="text-right">
                <div class="text-3xl font-bold text-blue-600">{{ $barragens->count() }}</div>
                <div class="text-sm text-gray-500">Barragens Monitoradas</div>
            </div>
        </div>
    </div>

    <!-- Filtros de Visualização -->
    <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
        <div class="bg-gradient-to-r from-blue-50 to-blue-100 border-b border-gray-200 px-6 py-4">
            <div class="flex items-center justify-between">
                <div class="flex items-center">
                    <i class="fas fa-filter text-blue-600 mr-3"></i>
                    <h2 class="text-lg font-semibold text-gray-800">Filtros de Visualização</h2>
                </div>
                <button onclick="atualizarGraficos()"
                        class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg text-sm font-medium flex items-center transition-all duration-200 shadow-sm hover:shadow-md">
                    <i class="fas fa-sync mr-2"></i>
                    Atualizar Gráficos
                </button>
            </div>
        </div>

        <div class="p-6">
            <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                <div class="space-y-2">
                    <label class="block text-sm font-medium text-gray-700 flex items-center">
                        <i class="fas fa-calendar-alt text-blue-500 mr-2"></i>
                        Período de Análise
                    </label>
                    <select id="periodo" class="w-full border border-gray-300 rounded-lg px-4 py-3 text-sm focus:border-blue-500 focus:ring-2 focus:ring-blue-200 bg-white shadow-sm transition-all duration-200">
                        <option value="7">Últimos 7 dias</option>
                        <option value="30" selected>Últimos 30 dias</option>
                        <option value="90">Últimos 3 meses</option>
                        <option value="365">Último ano</option>
                    </select>
                </div>

                <div class="space-y-2">
                    <label class="block text-sm font-medium text-gray-700 flex items-center">
                        <i class="fas fa-water text-blue-500 mr-2"></i>
                        Barragem Específica
                    </label>
                    <select id="barragem_filter" class="w-full border border-gray-300 rounded-lg px-4 py-3 text-sm focus:border-blue-500 focus:ring-2 focus:ring-blue-200 bg-white shadow-sm transition-all duration-200">
                        <option value="">Todas as Barragens</option>
                        @foreach($barragens as $barragem)
                            <option value="{{ $barragem->id }}">{{ $barragem->nome }}</option>
                        @endforeach
                    </select>
                </div>

                <div class="space-y-2">
                    <label class="block text-sm font-medium text-gray-700 flex items-center">
                        <i class="fas fa-chart-line text-blue-500 mr-2"></i>
                        Métrica de Visualização
                    </label>
                    <select id="metrica" class="w-full border border-gray-300 rounded-lg px-4 py-3 text-sm focus:border-blue-500 focus:ring-2 focus:ring-blue-200 bg-white shadow-sm transition-all duration-200">
                        <option value="percentagem_enchimento">Percentagem de Enchimento (%)</option>
                        <option value="cota_actual">Cota Atual (m)</option>
                        <option value="volume_actual">Volume Atual (Mm³)</option>
                    </select>
                </div>
            </div>

            <div class="mt-4 pt-4 border-t border-gray-100">
                <div class="flex items-center justify-between text-sm text-gray-600">
                    <div class="flex items-center">
                        <i class="fas fa-info-circle mr-2"></i>
                        <span>Selecione os filtros desejados e clique em "Atualizar Gráficos" para visualizar os dados</span>
                    </div>
                    <div class="flex items-center space-x-4">
                        <button onclick="limparFiltros()" class="text-gray-500 hover:text-gray-700 flex items-center">
                            <i class="fas fa-undo mr-1"></i>
                            Limpar
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Resumo das Barragens -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        @php
            $totalBarragens = $barragens->count();
            $barragensComLeituras = $barragens->filter(fn($b) => $b->ultima_leitura)->count();
            $mediaEnchimento = $barragens->filter(fn($b) => $b->ultima_leitura)->avg(fn($b) => $b->ultima_leitura->percentagem_enchimento ?? 0);
            $barragensAlerta = $barragens->filter(function($b) {
                return $b->ultima_leitura && $b->ultima_leitura->cota_actual >= $b->cota_alerta;
            })->count();
        @endphp

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-water text-blue-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Total de Barragens</p>
                    <p class="text-2xl font-bold text-gray-900">{{ $totalBarragens }}</p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-chart-line text-green-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Com Leituras</p>
                    <p class="text-2xl font-bold text-gray-900">{{ $barragensComLeituras }}</p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-orange-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-percentage text-orange-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Enchimento Médio</p>
                    <p class="text-2xl font-bold text-gray-900">{{ number_format($mediaEnchimento, 1) }}%</p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 {{ $barragensAlerta > 0 ? 'bg-red-100' : 'bg-gray-100' }} rounded-lg flex items-center justify-center">
                        <i class="fas fa-exclamation-triangle {{ $barragensAlerta > 0 ? 'text-red-600' : 'text-gray-600' }} text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <p class="text-sm text-gray-500">Em Alerta</p>
                    <p class="text-2xl font-bold {{ $barragensAlerta > 0 ? 'text-red-600' : 'text-gray-900' }}">{{ $barragensAlerta }}</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Gráfico Principal -->
    <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between mb-6">
            <h2 class="text-lg font-semibold text-gray-800">Evolução Temporal</h2>
            <div class="flex items-center space-x-2">
                <span class="text-sm text-gray-500">Última atualização:</span>
                <span class="text-sm font-medium text-gray-700" id="last-update">{{ now()->format('d/m/Y H:i') }}</span>
            </div>
        </div>
        <div class="h-96">
            <canvas id="grafico-principal"></canvas>
        </div>
    </div>

    <!-- Lista de Barragens -->
    <div class="bg-white rounded-lg shadow">
        <div class="px-6 py-4 border-b border-gray-200">
            <h2 class="text-lg font-semibold text-gray-800">Estado Atual das Barragens</h2>
        </div>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Barragem</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Última Leitura</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Cota Atual</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Volume Atual</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Enchimento</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Ações</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @foreach($barragens as $barragem)
                    <tr class="hover:bg-gray-50">
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div class="flex items-center">
                                <div class="flex-shrink-0 h-8 w-8">
                                    <div class="h-8 w-8 rounded-full bg-blue-100 flex items-center justify-center">
                                        <i class="fas fa-water text-blue-600 text-xs"></i>
                                    </div>
                                </div>
                                <div class="ml-3">
                                    <div class="text-sm font-medium text-gray-900">{{ $barragem->nome }}</div>
                                    <div class="text-sm text-gray-500">{{ $barragem->codigo }}</div>
                                </div>
                            </div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            @if($barragem->ultima_leitura)
                                {{ $barragem->ultima_leitura->data_leitura->format('d/m/Y') }}
                            @else
                                <span class="text-gray-400">Sem dados</span>
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            @if($barragem->ultima_leitura)
                                {{ number_format($barragem->ultima_leitura->cota_actual, 2) }} m
                            @else
                                <span class="text-gray-400">-</span>
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            @if($barragem->ultima_leitura)
                                {{ number_format($barragem->ultima_leitura->volume_actual, 2) }} Mm³
                            @else
                                <span class="text-gray-400">-</span>
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            @if($barragem->ultima_leitura)
                                <div class="flex items-center">
                                    <div class="w-16 bg-gray-200 rounded-full h-2 mr-2">
                                        <div class="bg-blue-600 h-2 rounded-full"
                                             style="width: {{ min($barragem->ultima_leitura->percentagem_enchimento, 100) }}%"></div>
                                    </div>
                                    <span class="text-sm font-medium">{{ number_format($barragem->ultima_leitura->percentagem_enchimento, 1) }}%</span>
                                </div>
                            @else
                                <span class="text-gray-400">-</span>
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            @if($barragem->ultima_leitura)
                                @php
                                    $status = $barragem->getStatusNivel($barragem->ultima_leitura->cota_actual);
                                @endphp
                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium
                                    @if($status == 'normal') bg-green-100 text-green-800
                                    @elseif($status == 'alerta') bg-yellow-100 text-yellow-800
                                    @else bg-red-100 text-red-800 @endif">
                                    {{ ucfirst($status) }}
                                </span>
                            @else
                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
                                    Sem dados
                                </span>
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium space-x-2">
                            <a href="{{ route('charts.barragem-detail', $barragem) }}"
                               class="text-blue-600 hover:text-blue-900">
                                <i class="fas fa-chart-area mr-1"></i>Gráfico
                            </a>
                            <a href="{{ route('barragens.show', $barragem) }}"
                               class="text-gray-600 hover:text-gray-900">
                                <i class="fas fa-eye mr-1"></i>Detalhes
                            </a>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Scripts -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
let grafico = null;
const cores = [
    '#3B82F6', '#10B981', '#F59E0B', '#EF4444', '#8B5CF6',
    '#06B6D4', '#F97316', '#84CC16', '#EC4899', '#6366F1'
];

// Inicializar gráfico
document.addEventListener('DOMContentLoaded', function() {
    atualizarGraficos();
});

async function atualizarGraficos() {
    const periodo = document.getElementById('periodo').value;
    const barragemId = document.getElementById('barragem_filter').value;
    const metrica = document.getElementById('metrica').value;

    try {
        let data;
        if (barragemId) {
            // Gráfico de uma barragem específica
            const response = await fetch(`/graficos/api/barragem/${barragemId}/data?days=${periodo}`);
            data = await response.json();
            criarGraficoUnicoBarragem(data, metrica, barragemId);
        } else {
            // Gráfico comparativo de todas as barragens
            criarGraficoComparativo(metrica, periodo);
        }

        document.getElementById('last-update').textContent = new Date().toLocaleString('pt-PT', {
            day: '2-digit',
            month: '2-digit',
            year: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    } catch (error) {
        console.error('Erro ao carregar dados:', error);
    }
}

function criarGraficoUnicoBarragem(data, metrica, barragemId) {
    const ctx = document.getElementById('grafico-principal').getContext('2d');

    if (grafico) {
        grafico.destroy();
    }

    const labels = data.map(d => new Date(d.data_leitura).toLocaleDateString('pt-PT'));
    const valores = data.map(d => parseFloat(d[metrica]) || 0);

    grafico = new Chart(ctx, {
        type: 'line',
        data: {
            labels: labels,
            datasets: [{
                label: getMetricaLabel(metrica),
                data: valores,
                borderColor: cores[0],
                backgroundColor: cores[0] + '20',
                fill: true,
                tension: 0.4
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                title: {
                    display: true,
                    text: `${getMetricaLabel(metrica)} - Barragem Selecionada`
                },
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    beginAtZero: metrica === 'percentagem_enchimento' ? false : true,
                    max: metrica === 'percentagem_enchimento' ? 100 : undefined,
                    ticks: {
                        callback: function(value) {
                            return metrica === 'percentagem_enchimento' ? value + '%' :
                                   metrica === 'volume_actual' ? value + ' Mm³' :
                                   value + ' m';
                        }
                    }
                }
            }
        }
    });
}

async function criarGraficoComparativo(metrica, periodo) {
    const ctx = document.getElementById('grafico-principal').getContext('2d');

    if (grafico) {
        grafico.destroy();
    }

    // Para gráfico comparativo, vamos usar os dados atuais das barragens
    const barragens = @json($barragens->values());

    const labels = barragens.map(b => b.nome);
    const valores = barragens.map(b => {
        if (b.ultima_leitura) {
            const campo = metrica || 'percentagem_enchimento';
            return parseFloat(b.ultima_leitura[campo]) || 0;
        }
        return 0;
    });

    grafico = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [{
                label: getMetricaLabel(metrica),
                data: valores,
                backgroundColor: cores.slice(0, barragens.length),
                borderColor: cores.slice(0, barragens.length),
                borderWidth: 1
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                title: {
                    display: true,
                    text: `Comparação: ${getMetricaLabel(metrica)}`
                }
            },
            scales: {
                y: {
                    beginAtZero: metrica === 'percentagem_enchimento' ? false : true,
                    max: metrica === 'percentagem_enchimento' ? 100 : undefined,
                    ticks: {
                        callback: function(value) {
                            return metrica === 'percentagem_enchimento' ? value + '%' :
                                   metrica === 'volume_actual' ? value + ' Mm³' :
                                   value + ' m';
                        }
                    }
                }
            }
        }
    });
}

function getMetricaLabel(metrica) {
    const labels = {
        'percentagem_enchimento': 'Percentagem de Enchimento (%)',
        'cota_actual': 'Cota Atual (m)',
        'volume_actual': 'Volume Atual (Mm³)'
    };
    return labels[metrica] || metrica;
}

function limparFiltros() {
    document.getElementById('periodo').value = '30';
    document.getElementById('barragem_filter').value = '';
    document.getElementById('metrica').value = 'percentagem_enchimento';
    atualizarGraficos();
}
</script>
@endsection