@props([
    'latitude',
    'longitude',
    'nome' => null,
    'tipo' => 'estacao',
    'id' => null,
    'size' => 'sm'
])

@php
    $sizeClasses = [
        'xs' => 'text-xs px-2 py-1',
        'sm' => 'text-sm px-3 py-2',
        'md' => 'text-base px-4 py-2',
        'lg' => 'text-lg px-5 py-3',
    ];
    $sizeClass = $sizeClasses[$size] ?? $sizeClasses['sm'];
@endphp

@if($latitude && $longitude)
    <div class="inline-flex items-center space-x-2">
        <!-- Coordenadas clicáveis -->
        <a href="{{ route('mapas.show', [
                'tipo' => $tipo,
                'id' => $id,
                'lat' => $latitude,
                'lng' => $longitude
            ]) }}"
           class="group inline-flex items-center {{ $sizeClass }} bg-blue-50 hover:bg-blue-100 text-blue-700 hover:text-blue-900 rounded-lg border border-blue-200 hover:border-blue-300 transition-all duration-200 font-mono"
           title="Clique para ver no mapa interativo">

            <i class="fas fa-map-marked-alt mr-2 text-blue-500 group-hover:text-blue-700 transition-colors"></i>

            <div class="flex flex-col leading-tight">
                <span class="font-semibold">{{ number_format($latitude, 6) }}°</span>
                <span class="font-semibold">{{ number_format($longitude, 6) }}°</span>
            </div>

            <i class="fas fa-external-link-alt ml-2 text-xs text-blue-400 group-hover:text-blue-600 transition-colors"></i>
        </a>

        <!-- Ações adicionais -->
        <div class="flex items-center space-x-1">
            <!-- Copiar coordenadas -->
            <button onclick="copiarCoordenadas('{{ $latitude }}, {{ $longitude }}')"
                    class="p-2 text-gray-400 hover:text-gray-600 hover:bg-gray-100 rounded-lg transition-colors"
                    title="Copiar coordenadas">
                <i class="fas fa-copy text-xs"></i>
            </button>

            <!-- Abrir no Google Maps -->
            <a href="https://www.google.com/maps?q={{ $latitude }},{{ $longitude }}"
               target="_blank"
               class="p-2 text-gray-400 hover:text-green-600 hover:bg-green-50 rounded-lg transition-colors"
               title="Abrir no Google Maps">
                <i class="fab fa-google text-xs"></i>
            </a>

            <!-- Obter direções -->
            <a href="https://www.google.com/maps/dir/?api=1&destination={{ $latitude }},{{ $longitude }}"
               target="_blank"
               class="p-2 text-gray-400 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
               title="Obter direções">
                <i class="fas fa-route text-xs"></i>
            </a>
        </div>
    </div>

    @push('scripts')
    <script>
    function copiarCoordenadas(coordenadas) {
        if (navigator.clipboard) {
            navigator.clipboard.writeText(coordenadas).then(function() {
                // Mostrar feedback visual
                showToast('Coordenadas copiadas: ' + coordenadas, 'success');
            }, function(err) {
                console.error('Erro ao copiar: ', err);
                fallbackCopyTextToClipboard(coordenadas);
            });
        } else {
            fallbackCopyTextToClipboard(coordenadas);
        }
    }

    function fallbackCopyTextToClipboard(text) {
        var textArea = document.createElement("textarea");
        textArea.value = text;
        textArea.style.top = "0";
        textArea.style.left = "0";
        textArea.style.position = "fixed";

        document.body.appendChild(textArea);
        textArea.focus();
        textArea.select();

        try {
            var successful = document.execCommand('copy');
            if (successful) {
                showToast('Coordenadas copiadas: ' + text, 'success');
            } else {
                showToast('Erro ao copiar coordenadas', 'error');
            }
        } catch (err) {
            showToast('Erro ao copiar coordenadas', 'error');
        }

        document.body.removeChild(textArea);
    }

    function showToast(message, type = 'info') {
        // Remover toast anterior se existir
        const existingToast = document.getElementById('coordinates-toast');
        if (existingToast) {
            existingToast.remove();
        }

        // Criar novo toast
        const toast = document.createElement('div');
        toast.id = 'coordinates-toast';
        toast.className = `fixed bottom-4 right-4 z-50 px-4 py-3 rounded-lg shadow-lg transition-all duration-300 transform ${
            type === 'success' ? 'bg-green-100 border border-green-300 text-green-800' :
            type === 'error' ? 'bg-red-100 border border-red-300 text-red-800' :
            'bg-blue-100 border border-blue-300 text-blue-800'
        }`;

        toast.innerHTML = `
            <div class="flex items-center space-x-2">
                <i class="fas ${type === 'success' ? 'fa-check' : type === 'error' ? 'fa-times' : 'fa-info'} text-sm"></i>
                <span class="text-sm font-medium">${message}</span>
                <button onclick="this.parentElement.parentElement.remove()" class="ml-2 text-gray-500 hover:text-gray-700">
                    <i class="fas fa-times text-xs"></i>
                </button>
            </div>
        `;

        document.body.appendChild(toast);

        // Remover automaticamente após 3 segundos
        setTimeout(() => {
            if (toast && toast.parentNode) {
                toast.style.opacity = '0';
                toast.style.transform = 'translateY(100%)';
                setTimeout(() => {
                    if (toast && toast.parentNode) {
                        toast.remove();
                    }
                }, 300);
            }
        }, 3000);
    }
    </script>
    @endpush

@else
    <!-- Sem coordenadas -->
    <div class="inline-flex items-center {{ $sizeClass }} bg-gray-50 text-gray-500 rounded-lg border border-gray-200 font-mono">
        <i class="fas fa-map-marker-slash mr-2 text-gray-400"></i>
        <span class="text-gray-500">Sem coordenadas</span>
    </div>
@endif