<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Correspondências de Curvas - Sistema de Barragens</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body class="bg-gray-100">
    <div class="container mx-auto px-4 py-8">

        <!-- Header -->
        <div class="bg-gradient-to-r from-purple-600 to-indigo-600 rounded-lg shadow-lg text-white p-6 mb-6">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-3xl font-bold">CORRESPONDÊNCIAS DE CURVAS DE BARRAGENS</h1>
                    <p class="mt-2 opacity-90">Cálculo de correspondências entre Cota e Volume</p>
                </div>
                <div class="text-right">
                    <i class="fas fa-chart-area text-5xl opacity-75"></i>
                </div>
            </div>
        </div>

        <!-- Seleção de Barragem -->
        <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 mb-6">
            <h2 class="text-xl font-bold mb-4 flex items-center">
                <i class="fas fa-water mr-2 text-blue-500"></i>
                Selecionar Barragem
            </h2>

            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <label for="barragem-select" class="block text-sm font-medium text-gray-700 mb-2">
                        Escolha a barragem:
                    </label>
                    <select id="barragem-select" class="w-full border border-gray-300 rounded-lg px-4 py-2">
                        <option value="">-- Selecione uma barragem --</option>
                        @foreach($barragens as $barragem)
                            <option value="{{ $barragem->id }}">{{ $barragem->nome }}</option>
                        @endforeach
                    </select>
                </div>

                <div id="info-barragem" class="hidden">
                    <div class="bg-blue-50 p-4 rounded-lg">
                        <h3 class="font-bold text-blue-800 mb-2">Informações da Barragem</h3>
                        <div class="text-sm space-y-1">
                            <p><strong>Cota Mínima:</strong> <span id="info-cota-min">-</span> m</p>
                            <p><strong>Cota Máxima:</strong> <span id="info-cota-max">-</span> m</p>
                            <p><strong>Volume Máximo:</strong> <span id="info-volume-max">-</span> Mm³</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Calculadora de Correspondências -->
        <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">

            <!-- Calcular Volume a partir de Cota -->
            <div class="bg-white rounded-lg shadow-lg p-6">
                <h3 class="text-xl font-bold mb-4 flex items-center">
                    <i class="fas fa-arrow-right mr-2 text-green-500"></i>
                    Cota → Volume
                </h3>

                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">
                            Cota (m):
                        </label>
                        <input type="number" id="input-cota" step="0.01"
                               class="w-full border border-gray-300 rounded-lg px-4 py-2"
                               placeholder="Digite a cota em metros">
                    </div>

                    <button onclick="calcularVolume()"
                            class="w-full bg-green-500 text-white px-4 py-3 rounded-lg hover:bg-green-600 transition-colors font-bold">
                        <i class="fas fa-calculator mr-2"></i>Calcular Volume
                    </button>

                    <div id="resultado-volume" class="hidden bg-green-50 p-4 rounded-lg">
                        <h4 class="font-bold text-green-800 mb-2">Resultado:</h4>
                        <div class="space-y-2 text-sm">
                            <p><strong>Volume:</strong> <span id="res-volume" class="text-lg text-green-700">-</span> Mm³</p>
                            <p><strong>Enchimento:</strong> <span id="res-vol-percent" class="text-lg text-green-700">-</span>%</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Calcular Cota a partir de Volume -->
            <div class="bg-white rounded-lg shadow-lg p-6">
                <h3 class="text-xl font-bold mb-4 flex items-center">
                    <i class="fas fa-arrow-left mr-2 text-blue-500"></i>
                    Volume → Cota
                </h3>

                <div class="space-y-4">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">
                            Volume (Mm³):
                        </label>
                        <input type="number" id="input-volume" step="0.01"
                               class="w-full border border-gray-300 rounded-lg px-4 py-2"
                               placeholder="Digite o volume em Mm³">
                    </div>

                    <button onclick="calcularCota()"
                            class="w-full bg-blue-500 text-white px-4 py-3 rounded-lg hover:bg-blue-600 transition-colors font-bold">
                        <i class="fas fa-calculator mr-2"></i>Calcular Cota
                    </button>

                    <div id="resultado-cota" class="hidden bg-blue-50 p-4 rounded-lg">
                        <h4 class="font-bold text-blue-800 mb-2">Resultado:</h4>
                        <div class="space-y-2 text-sm">
                            <p><strong>Cota:</strong> <span id="res-cota" class="text-lg text-blue-700">-</span> m</p>
                            <p><strong>Enchimento:</strong> <span id="res-cota-percent" class="text-lg text-blue-700">-</span>%</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Tabela de Correspondências Completa -->
        <div class="bg-white rounded-lg shadow-lg p-6 mb-6">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-xl font-bold flex items-center">
                    <i class="fas fa-table mr-2 text-purple-500"></i>
                    Tabela de Correspondências
                </h3>
                <button onclick="gerarTabelaCompleta()"
                        class="bg-purple-500 text-white px-4 py-2 rounded-lg hover:bg-purple-600 transition-colors">
                    <i class="fas fa-sync mr-2"></i>Gerar Tabela
                </button>
            </div>

            <div id="tabela-container" class="overflow-x-auto hidden">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Cota (m)
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Volume (Mm³)
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Área Espelho d'Água (ha)
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Enchimento (%)
                            </th>
                        </tr>
                    </thead>
                    <tbody id="tabela-body" class="bg-white divide-y divide-gray-200">
                        <!-- Dados preenchidos via JavaScript -->
                    </tbody>
                </table>
            </div>

            <div id="tabela-vazia" class="text-center py-8 text-gray-500">
                <i class="fas fa-table text-4xl mb-2 opacity-50"></i>
                <p>Selecione uma barragem e clique em "Gerar Tabela"</p>
            </div>
        </div>

        <!-- Validação de Curvas -->
        <div class="bg-white rounded-lg shadow-lg p-6">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-xl font-bold flex items-center">
                    <i class="fas fa-check-circle mr-2 text-yellow-500"></i>
                    Validação de Curvas
                </h3>
                <button onclick="validarCurvas()"
                        class="bg-yellow-500 text-white px-4 py-2 rounded-lg hover:bg-yellow-600 transition-colors">
                    <i class="fas fa-clipboard-check mr-2"></i>Validar
                </button>
            </div>

            <div id="validacao-resultado" class="hidden">
                <!-- Resultados preenchidos via JavaScript -->
            </div>

            <div id="validacao-vazia" class="text-center py-8 text-gray-500">
                <i class="fas fa-clipboard-check text-4xl mb-2 opacity-50"></i>
                <p>Selecione uma barragem e clique em "Validar"</p>
            </div>
        </div>
    </div>

    <script>
        let barragemSelecionada = null;

        // Event listener para seleção de barragem
        document.getElementById('barragem-select').addEventListener('change', function() {
            const barragemId = this.value;

            if (!barragemId) {
                document.getElementById('info-barragem').classList.add('hidden');
                barragemSelecionada = null;
                return;
            }

            barragemSelecionada = barragemId;
            carregarDadosBarragem(barragemId);
        });

        function carregarDadosBarragem(barragemId) {
            fetch(`/api/correspondencias-curvas/barragem/${barragemId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.barragem) {
                        document.getElementById('info-cota-min').textContent = data.barragem.cota_minima || 'N/A';
                        document.getElementById('info-cota-max').textContent = data.barragem.cota_maxima || 'N/A';
                        document.getElementById('info-volume-max').textContent = data.barragem.volume_maximo || 'N/A';
                        document.getElementById('info-barragem').classList.remove('hidden');
                    }
                })
                .catch(error => {
                    console.error('Erro ao carregar dados da barragem:', error);
                    alert('Erro ao carregar dados da barragem');
                });
        }

        function calcularVolume() {
            if (!barragemSelecionada) {
                alert('Por favor, selecione uma barragem primeiro');
                return;
            }

            const cota = document.getElementById('input-cota').value;
            if (!cota) {
                alert('Por favor, digite uma cota');
                return;
            }

            fetch('/api/correspondencias-curvas/calcular', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({
                    barragem_id: barragemSelecionada,
                    tipo_calculo: 'cota_para_volume',
                    valor: parseFloat(cota)
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.sucesso) {
                    document.getElementById('res-volume').textContent = data.volume_calculado;
                    document.getElementById('res-vol-percent').textContent = data.percentagem_enchimento.toFixed(2);
                    document.getElementById('resultado-volume').classList.remove('hidden');
                } else {
                    alert('Erro ao calcular volume');
                }
            })
            .catch(error => {
                console.error('Erro:', error);
                alert('Erro ao realizar cálculo');
            });
        }

        function calcularCota() {
            if (!barragemSelecionada) {
                alert('Por favor, selecione uma barragem primeiro');
                return;
            }

            const volume = document.getElementById('input-volume').value;
            if (!volume) {
                alert('Por favor, digite um volume');
                return;
            }

            fetch('/api/correspondencias-curvas/calcular', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({
                    barragem_id: barragemSelecionada,
                    tipo_calculo: 'volume_para_cota',
                    valor: parseFloat(volume)
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.sucesso) {
                    document.getElementById('res-cota').textContent = data.cota_calculada;
                    document.getElementById('res-cota-percent').textContent = data.percentagem_enchimento.toFixed(2);
                    document.getElementById('resultado-cota').classList.remove('hidden');
                } else {
                    alert('Erro ao calcular cota');
                }
            })
            .catch(error => {
                console.error('Erro:', error);
                alert('Erro ao realizar cálculo');
            });
        }

        function gerarTabelaCompleta() {
            if (!barragemSelecionada) {
                alert('Por favor, selecione uma barragem primeiro');
                return;
            }

            fetch(`/api/correspondencias-curvas/tabela/${barragemSelecionada}`)
                .then(response => response.json())
                .then(data => {
                    if (data.sucesso) {
                        const tbody = document.getElementById('tabela-body');
                        tbody.innerHTML = '';

                        data.tabela_correspondencias.forEach(row => {
                            const tr = document.createElement('tr');
                            tr.innerHTML = `
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">${row.cota}</td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">${row.volume}</td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">${row.area_espelho_agua}</td>
                                <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">${row.percentagem_enchimento}%</td>
                            `;
                            tbody.appendChild(tr);
                        });

                        document.getElementById('tabela-container').classList.remove('hidden');
                        document.getElementById('tabela-vazia').classList.add('hidden');
                    } else {
                        alert(data.mensagem || 'Erro ao gerar tabela');
                    }
                })
                .catch(error => {
                    console.error('Erro:', error);
                    alert('Erro ao gerar tabela');
                });
        }

        function validarCurvas() {
            if (!barragemSelecionada) {
                alert('Por favor, selecione uma barragem primeiro');
                return;
            }

            fetch(`/api/correspondencias-curvas/validar/${barragemSelecionada}`)
                .then(response => response.json())
                .then(data => {
                    const resultadoDiv = document.getElementById('validacao-resultado');
                    resultadoDiv.innerHTML = '';

                    // Status geral
                    const statusClass = data.sucesso ? 'bg-green-50 border-green-200' : 'bg-red-50 border-red-200';
                    const statusIcon = data.sucesso ? 'fa-check-circle text-green-500' : 'fa-exclamation-triangle text-red-500';
                    const statusText = data.sucesso ? 'Curvas Válidas' : 'Problemas Encontrados';

                    let html = `
                        <div class="p-4 ${statusClass} border rounded-lg mb-4">
                            <h4 class="font-bold flex items-center mb-2">
                                <i class="fas ${statusIcon} mr-2"></i>
                                ${statusText}
                            </h4>
                            <p class="text-sm">Total de pontos: ${data.total_pontos || 0}</p>
                        </div>
                    `;

                    // Erros
                    if (data.erros && data.erros.length > 0) {
                        html += '<div class="mb-4"><h5 class="font-bold text-red-700 mb-2">❌ Erros:</h5><ul class="list-disc list-inside space-y-1">';
                        data.erros.forEach(erro => {
                            html += `<li class="text-sm text-red-600">${erro}</li>`;
                        });
                        html += '</ul></div>';
                    }

                    // Avisos
                    if (data.avisos && data.avisos.length > 0) {
                        html += '<div class="mb-4"><h5 class="font-bold text-yellow-700 mb-2">⚠️ Avisos:</h5><ul class="list-disc list-inside space-y-1">';
                        data.avisos.forEach(aviso => {
                            html += `<li class="text-sm text-yellow-600">${aviso}</li>`;
                        });
                        html += '</ul></div>';
                    }

                    // Intervalos
                    if (data.intervalo_cotas && data.intervalo_volumes) {
                        html += `
                            <div class="grid grid-cols-2 gap-4 mt-4">
                                <div class="bg-blue-50 p-3 rounded">
                                    <h5 class="font-bold text-blue-800 text-sm mb-1">Intervalo de Cotas</h5>
                                    <p class="text-sm">${data.intervalo_cotas.minima}m - ${data.intervalo_cotas.maxima}m</p>
                                </div>
                                <div class="bg-purple-50 p-3 rounded">
                                    <h5 class="font-bold text-purple-800 text-sm mb-1">Intervalo de Volumes</h5>
                                    <p class="text-sm">${data.intervalo_volumes.minimo} - ${data.intervalo_volumes.maximo} Mm³</p>
                                </div>
                            </div>
                        `;
                    }

                    resultadoDiv.innerHTML = html;
                    resultadoDiv.classList.remove('hidden');
                    document.getElementById('validacao-vazia').classList.add('hidden');
                })
                .catch(error => {
                    console.error('Erro:', error);
                    alert('Erro ao validar curvas');
                });
        }
    </script>
</body>
</html>
