@extends('layouts.app')

@section('title', 'Dashboard - Sistema Hidrológico')

@section('content')
<div class="space-y-6">
    <!-- Dashboard Superior - Sistema Hidrológico ARA Norte -->
    <div class="bg-white rounded-lg shadow">
        <div class="px-6 py-4 border-b border-gray-200 bg-gradient-to-r from-blue-50 to-indigo-50">
            <div class="flex items-center justify-between">
                <div>
                    <h1 class="text-2xl font-bold text-gray-900">
                        <i class="fas fa-chart-line mr-3 text-blue-600"></i>
                        Dashboard - Sistema Hidrológico ARA Norte
                    </h1>
                    <p class="text-gray-600 mt-1">Monitoramento integrado de recursos hídricos</p>
                </div>
                <div class="text-right">
                    <div class="text-sm text-gray-600 mb-1">Última atualização</div>
                    <div class="text-sm font-mono text-gray-800">{{ now()->format('d/m/Y H:i') }}</div>
                </div>
            </div>
        </div>

        <!-- Cards de Informações do Ano Hidrológico -->
        <div class="px-6 py-4">
            <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
                <!-- Ano Hidrológico Atual -->
                <div class="bg-white border border-gray-200 rounded-lg shadow-sm hover:shadow-md transition-shadow duration-200 p-6">
                    <div class="flex items-center justify-between mb-4">
                        <div class="flex items-center space-x-3">
                            <div class="w-10 h-10 bg-blue-100 rounded-lg flex items-center justify-center">
                                <i class="fas fa-calendar-alt text-blue-600 text-lg"></i>
                            </div>
                            <div>
                                <h3 class="text-lg font-semibold text-gray-900">Ano Hidrológico</h3>
                                <p class="text-sm text-gray-500">Período de gestão</p>
                            </div>
                        </div>
                    </div>

                    <div class="mb-4">
                        <div class="text-2xl font-bold text-gray-900">
                            {{ $stats['ano_hidrologico']['label'] }}
                        </div>
                        <div class="text-sm text-gray-600 mt-1">
                            {{ $stats['ano_hidrologico']['start']->format('d M Y') }} - {{ $stats['ano_hidrologico']['end']->format('d M Y') }}
                        </div>
                    </div>

                    <div class="space-y-2">
                        <div class="flex justify-between text-sm">
                            <span class="text-gray-600">Progresso</span>
                            <span class="font-medium text-gray-900">{{ $stats['progresso_ano'] }}%</span>
                        </div>
                        <div class="w-full bg-gray-200 rounded-full h-2">
                            <div class="bg-blue-600 h-2 rounded-full transition-all duration-300"
                                 style="width: {{ $stats['progresso_ano'] }}%"></div>
                        </div>
                    </div>
                </div>

                <!-- Estação Atual -->
                <div class="bg-white border border-gray-200 rounded-lg shadow-sm hover:shadow-md transition-shadow duration-200 p-6">
                    <div class="flex items-center justify-between mb-4">
                        <div class="flex items-center space-x-3">
                            <div class="w-10 h-10 bg-{{ $stats['estacao_atual']['color'] }}-100 rounded-lg flex items-center justify-center">
                                @if($stats['estacao_atual']['season'] == 'chuvosa')
                                    <i class="fas fa-cloud-rain text-{{ $stats['estacao_atual']['color'] }}-600 text-lg"></i>
                                @else
                                    <i class="fas fa-sun text-{{ $stats['estacao_atual']['color'] }}-600 text-lg"></i>
                                @endif
                            </div>
                            <div>
                                <h3 class="text-lg font-semibold text-gray-900">{{ $stats['estacao_atual']['label'] }}</h3>
                                <p class="text-sm text-gray-500">Estado climático</p>
                            </div>
                        </div>
                        <div class="px-2 py-1 bg-{{ $stats['estacao_atual']['color'] }}-100 text-{{ $stats['estacao_atual']['color'] }}-700 text-xs font-medium rounded">
                            ATIVO
                        </div>
                    </div>

                    <div class="mb-4">
                        <div class="text-xl font-bold text-gray-900">
                            {{ $stats['estacao_atual']['months'] }}
                        </div>
                        <div class="text-sm text-gray-600 mt-1">
                            Período atual do ano hidrológico
                        </div>
                    </div>
                </div>

                <!-- Leituras do Ano Hidrológico -->
                <div class="bg-white border border-gray-200 rounded-lg shadow-sm hover:shadow-md transition-shadow duration-200 p-6">
                    <div class="flex items-center justify-between mb-4">
                        <div class="flex items-center space-x-3">
                            <div class="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center">
                                <i class="fas fa-chart-bar text-green-600 text-lg"></i>
                            </div>
                            <div>
                                <h3 class="text-lg font-semibold text-gray-900">Leituras do Ano</h3>
                                <p class="text-sm text-gray-500">Total coletado</p>
                            </div>
                        </div>
                    </div>

                    <div class="mb-4">
                        <div class="text-2xl font-bold text-gray-900">
                            {{ number_format($stats['leituras_ano_hidrologico'] ?? 0) }}
                        </div>
                        <div class="text-sm text-gray-600 mt-1">
                            Registros no período
                        </div>
                    </div>
                </div>
            </div>

            <!-- Estatísticas Principais -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-6">
                <!-- Card Barragens -->
                <div class="bg-white border border-gray-200 rounded-lg shadow-sm hover:shadow-md transition-shadow duration-200 p-6">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-3">
                            <div class="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                                <i class="fas fa-water text-blue-600 text-xl"></i>
                            </div>
                            <div>
                                <h3 class="text-lg font-semibold text-gray-900">Barragens</h3>
                                <p class="text-sm text-gray-600">Sistema de reserva</p>
                            </div>
                        </div>
                        <div class="text-right">
                            <div class="text-3xl font-bold text-gray-900">{{ $stats['total_barragens'] ?? 6 }}</div>
                        </div>
                    </div>
                </div>

                <!-- Card Estações -->
                <div class="bg-white border border-gray-200 rounded-lg shadow-sm hover:shadow-md transition-shadow duration-200 p-6">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-3">
                            <div class="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center">
                                <i class="fas fa-satellite-dish text-green-600 text-xl"></i>
                            </div>
                            <div>
                                <h3 class="text-lg font-semibold text-gray-900">Estações</h3>
                                <p class="text-sm text-gray-600">Rede de monitoramento</p>
                            </div>
                        </div>
                        <div class="text-right">
                            <div class="text-3xl font-bold text-gray-900">{{ $stats['total_estacoes'] ?? 15 }}</div>
                        </div>
                    </div>
                </div>

                <!-- Card Alertas -->
                <div class="bg-white border border-gray-200 rounded-lg shadow-sm hover:shadow-md transition-shadow duration-200 p-6">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-3">
                            <div class="w-12 h-12 bg-red-100 rounded-lg flex items-center justify-center">
                                <i class="fas fa-triangle-exclamation text-red-600 text-xl"></i>
                            </div>
                            <div>
                                <h3 class="text-lg font-semibold text-gray-900">Alertas</h3>
                                <p class="text-sm text-gray-600">Sistema de avisos</p>
                            </div>
                        </div>
                        <div class="text-right">
                            <div class="text-3xl font-bold text-gray-900">{{ $stats['alertas_ativos'] ?? 2 }}</div>
                        </div>
                    </div>
                </div>

                <!-- Card Leituras -->
                <div class="bg-white border border-gray-200 rounded-lg shadow-sm hover:shadow-md transition-shadow duration-200 p-6">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center space-x-3">
                            <div class="w-12 h-12 bg-purple-100 rounded-lg flex items-center justify-center">
                                <i class="fas fa-chart-line text-purple-600 text-xl"></i>
                            </div>
                            <div>
                                <h3 class="text-lg font-semibold text-gray-900">Leituras Hoje</h3>
                                <p class="text-sm text-gray-600">Coleta automatizada</p>
                            </div>
                        </div>
                        <div class="text-right">
                            <div class="text-3xl font-bold text-gray-900">{{ number_format($stats['leituras_hoje'] ?? 0) }}</div>
                        </div>
                    </div>
                </div>
            </div>

        </div>
    </div>

    <!-- Filtros e Controles do Mapa GIS -->
    <div class="bg-white rounded-lg shadow p-5">
        <div class="flex items-center justify-between mb-4">
            <h3 class="text-lg font-semibold text-gray-800">
                <i class="fas fa-layer-group mr-2 text-blue-600"></i>
                Controles de Filtros GIS
            </h3>
            <div class="bg-blue-600 px-3 py-1 rounded text-sm font-medium text-white">
                <i class="fas fa-satellite-dish mr-1"></i>Sistema ARA Norte
            </div>
        </div>

        <!-- Filtros Simplificados -->
        <div class="bg-gradient-to-r from-gray-50 to-blue-50 rounded-lg p-4 mb-4">
            <div class="grid grid-cols-1 md:grid-cols-5 gap-4">
                <!-- Filtro de Bacia Hidrográfica -->
                <div>
                    <label class="block text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-water mr-1 text-blue-500"></i>
                        Bacia Hidrográfica
                    </label>
                    <select id="filter-bacia-hidro" class="w-full px-3 py-2 text-sm border border-gray-300 rounded-lg focus:border-blue-500 focus:ring-2 focus:ring-blue-200 bg-white">
                        <option value="">Todas as bacias</option>
                        <option value="bacia_rovuma">Bacia Rovuma</option>
                        <option value="bacia_messalo">Bacia Messalo</option>
                        <option value="bacia_licungo">Bacia Lúrio</option>
                        <option value="bacia_monapo">Bacia Monapo</option>
                        <option value="bacia_meluli">Bacia Meluli</option>
                        <option value="bacia_mecuburi">Bacia Mecuburi</option>
                        <option value="megarruma">Bacia Megarruma</option>
                    </select>
                </div>

                <!-- Filtro de Província -->
                <div>
                    <label class="block text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-map-marker-alt mr-1 text-green-500"></i>
                        Província
                    </label>
                    <select id="filter-provincia" class="w-full px-3 py-2 text-sm border border-gray-300 rounded-lg focus:border-green-500 focus:ring-2 focus:ring-green-200 bg-white">
                        <option value="">Todas as províncias</option>
                        <option value="Cabo Delgado">Cabo Delgado</option>
                        <option value="Nampula">Nampula</option>
                        <option value="Niassa">Niassa</option>
                        <option value="Zambézia">Zambézia</option>
                        <option value="Tete">Tete</option>
                    </select>
                </div>

                <!-- Filtro de Tipo de Infraestrutura -->
                <div>
                    <label class="block text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-cogs mr-1 text-orange-500"></i>
                        Infraestrutura
                    </label>
                    <select id="filter-infraestrutura-tipo" class="w-full px-3 py-2 text-sm border border-gray-300 rounded-lg focus:border-orange-500 focus:ring-2 focus:ring-orange-200 bg-white">
                        <option value="">Todos os tipos</option>
                        <option value="barragens">Barragens</option>
                        <option value="albufeiras">Albufeiras</option>
                        <option value="rios">Rios</option>
                        <option value="divisores">Divisores</option>
                    </select>
                </div>

                <!-- Filtro de Tipo de Estação -->
                <div>
                    <label class="block text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-broadcast-tower mr-1 text-purple-500"></i>
                        Tipo de Estação
                    </label>
                    <select id="filter-tipo-estacao" class="w-full px-3 py-2 text-sm border border-gray-300 rounded-lg focus:border-purple-500 focus:ring-2 focus:ring-purple-200 bg-white">
                        <option value="">Todos os tipos</option>
                        <option value="hidrometrica">Hidrométricas</option>
                        <option value="pluviometrica">Pluviométricas</option>
                        <option value="meteorologica">Meteorológicas</option>
                        <option value="climatologica">Climatológicas</option>
                    </select>
                </div>

                <!-- Pesquisa por nome -->
                <div>
                    <label class="block text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-search mr-1 text-gray-500"></i>
                        Pesquisar
                    </label>
                    <input type="text" id="filter-search" placeholder="Nome ou código..." class="w-full px-3 py-2 text-sm border border-gray-300 rounded-lg focus:border-gray-500 focus:ring-2 focus:ring-gray-200 bg-white">
                </div>
            </div>
        </div>

        <!-- Controles de Ação -->
        <div class="flex flex-wrap gap-3 justify-between items-center">
            <div class="flex flex-wrap gap-3">
                <button id="btn-aplicar-filtros" style="background: linear-gradient(45deg, #10b981, #059669); color: white; padding: 12px 20px; border-radius: 8px; border: 2px solid #059669; font-weight: 600; box-shadow: 0 4px 6px rgba(0,0,0,0.1); cursor: pointer; display: inline-flex; align-items: center; min-width: 150px; margin: 5px;">
                    <i class="fas fa-check" style="margin-right: 8px;"></i>Aplicar Filtros
                </button>
                <button id="btn-limpar-filtros" style="background: linear-gradient(45deg, #6b7280, #4b5563); color: white; padding: 12px 20px; border-radius: 8px; border: 2px solid #4b5563; font-weight: 600; box-shadow: 0 4px 6px rgba(0,0,0,0.1); cursor: pointer; display: inline-flex; align-items: center; min-width: 150px; margin: 5px;">
                    <i class="fas fa-eraser" style="margin-right: 8px;"></i>Limpar Tudo
                </button>
                <button id="btn-mostrar-tudo" style="background: linear-gradient(45deg, #3b82f6, #2563eb); color: white; padding: 12px 20px; border-radius: 8px; border: 2px solid #2563eb; font-weight: 600; box-shadow: 0 4px 6px rgba(0,0,0,0.1); cursor: pointer; display: inline-flex; align-items: center; min-width: 150px; margin: 5px;">
                    <i class="fas fa-eye" style="margin-right: 8px;"></i>Mostrar Tudo
                </button>
            </div>

            <!-- Status em Tempo Real -->
            <div class="bg-blue-50 border-l-4 border-blue-500 px-3 py-2 rounded-r">
                <div id="filter-status" class="text-sm font-medium text-gray-700">
                    <i class="fas fa-info-circle mr-1 text-blue-500"></i>
                    📊 Todas as camadas visíveis
                </div>
            </div>
        </div>
    </div>

    <!-- Map Container -->
    <div class="bg-white rounded-lg shadow">
        <div class="p-4 border-b bg-gradient-to-r from-blue-50 to-green-50">
            <div class="flex justify-between items-center">
                <h3 class="text-lg font-medium text-gray-900">
                    <i class="fas fa-globe-africa mr-2 text-blue-600"></i>
                    Mapa GIS - Sistema Hidrológico
                </h3>
                <div class="flex items-center space-x-4">
                    <span class="text-sm text-gray-600">Coordenadas: <span id="mouse-coords">--, --</span></span>
                    <span class="text-sm text-gray-600">Zoom: <span id="current-zoom">7</span></span>
                </div>
            </div>
        </div>
        <div class="relative" id="map-container">
            <!-- Painel Lateral de Camadas Estilo SIRHAN -->
            <div id="layers-panel" class="absolute left-4 top-4 bottom-4 w-80 bg-white rounded-lg shadow-2xl z-[1000] overflow-hidden flex flex-col">
                <!-- Header do Painel -->
                <div class="bg-gradient-to-r from-blue-600 to-blue-700 px-4 py-3 flex items-center justify-between">
                    <h3 class="text-white font-bold text-lg flex items-center">
                        <i class="fas fa-layer-group mr-2"></i>
                        Camadas GIS
                    </h3>
                    <button id="toggle-layers-panel" class="text-white hover:bg-blue-800 rounded p-1 transition-colors">
                        <i class="fas fa-chevron-left"></i>
                    </button>
                </div>

                <!-- Conte\u00fado do Painel com Scroll -->
                <div class="flex-1 overflow-y-auto p-4 space-y-4">
                    <!-- Grupo: Mapas Base -->
                    <div class="border border-gray-200 rounded-lg overflow-hidden">
                        <div class="bg-gray-100 px-3 py-2 font-semibold text-gray-700 text-sm flex items-center justify-between cursor-pointer" onclick="toggleGroup('base-maps')">
                            <span><i class="fas fa-map mr-2 text-gray-600"></i>Mapas Base</span>
                            <i class="fas fa-chevron-down text-gray-500" id="icon-base-maps"></i>
                        </div>
                        <div id="group-base-maps" class="p-3 space-y-2 bg-white">
                            <label class="flex items-center space-x-2 cursor-pointer hover:bg-gray-50 p-2 rounded">
                                <input type="radio" name="base-layer" value="topo" class="form-radio text-blue-600" checked>
                                <span class="text-sm">Topogr\u00e1fico</span>
                            </label>
                            <label class="flex items-center space-x-2 cursor-pointer hover:bg-gray-50 p-2 rounded">
                                <input type="radio" name="base-layer" value="satellite" class="form-radio text-blue-600">
                                <span class="text-sm">Sat\u00e9lite</span>
                            </label>
                            <label class="flex items-center space-x-2 cursor-pointer hover:bg-gray-50 p-2 rounded">
                                <input type="radio" name="base-layer" value="standard" class="form-radio text-blue-600">
                                <span class="text-sm">Padr\u00e3o</span>
                            </label>
                        </div>
                    </div>

                    <!-- Grupo: Bacias Hidrogr\u00e1ficas -->
                    <div class="border border-gray-200 rounded-lg overflow-hidden">
                        <div class="bg-blue-50 px-3 py-2 font-semibold text-blue-800 text-sm flex items-center justify-between cursor-pointer" onclick="toggleGroup('bacias')">
                            <span><i class="fas fa-water mr-2"></i>Bacias Hidrogr\u00e1ficas</span>
                            <i class="fas fa-chevron-down text-blue-600" id="icon-bacias"></i>
                        </div>
                        <div id="group-bacias" class="p-3 bg-white">
                            <!-- Aviso sobre shapefiles -->
                            <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-3 mb-3">
                                <div class="flex items-start space-x-2">
                                    <i class="fas fa-exclamation-triangle text-yellow-600 mt-1 text-sm"></i>
                                    <div class="text-xs text-yellow-800">
                                        <strong>Shapefiles Genéricos</strong>
                                        <p class="mt-1">Os arquivos das bacias são idênticos. Para ver as delimitações reais, substitua os .shp em <code class="bg-yellow-100 px-1 rounded text-xs">/public/shapes/</code> pelos dados oficiais da ARA Norte.</p>
                                    </div>
                                </div>
                            </div>

                            <div class="space-y-2 opacity-50">
                                <label class="flex items-center space-x-2 cursor-not-allowed p-2 rounded" title="Shapefiles com dados genéricos - desabilitado">
                                    <input type="checkbox" class="form-checkbox text-red-600" disabled>
                                    <div class="w-4 h-4 rounded" style="background: #E74C3C;"></div>
                                    <span class="text-sm text-gray-600">Bacia Rovuma</span>
                                </label>
                                <label class="flex items-center space-x-2 cursor-not-allowed p-2 rounded" title="Shapefiles com dados genéricos - desabilitado">
                                    <input type="checkbox" class="form-checkbox text-green-600" disabled>
                                    <div class="w-4 h-4 rounded" style="background: #2ECC71;"></div>
                                    <span class="text-sm text-gray-600">Bacia Messalo</span>
                                </label>
                                <label class="flex items-center space-x-2 cursor-not-allowed p-2 rounded" title="Shapefiles com dados genéricos - desabilitado">
                                    <input type="checkbox" class="form-checkbox text-blue-600" disabled>
                                    <div class="w-4 h-4 rounded" style="background: #3498DB;"></div>
                                    <span class="text-sm text-gray-600">Bacia Lúrio</span>
                                </label>
                                <label class="flex items-center space-x-2 cursor-not-allowed p-2 rounded" title="Shapefiles com dados genéricos - desabilitado">
                                    <input type="checkbox" class="form-checkbox text-orange-600" disabled>
                                    <div class="w-4 h-4 rounded" style="background: #F39C12;"></div>
                                    <span class="text-sm text-gray-600">Bacia Monapo</span>
                                </label>
                            </div>
                        </div>
                    </div>

                    <!-- Grupo: Esta\u00e7\u00f5es -->
                    <div class="border border-gray-200 rounded-lg overflow-hidden">
                        <div class="bg-green-50 px-3 py-2 font-semibold text-green-800 text-sm flex items-center justify-between cursor-pointer" onclick="toggleGroup('estacoes')">
                            <span><i class="fas fa-broadcast-tower mr-2"></i>Esta\u00e7\u00f5es</span>
                            <i class="fas fa-chevron-down text-green-600" id="icon-estacoes"></i>
                        </div>
                        <div id="group-estacoes" class="p-3 space-y-2 bg-white">
                            <label class="flex items-center space-x-2 cursor-pointer hover:bg-green-50 p-2 rounded">
                                <input type="checkbox" class="marker-toggle form-checkbox text-green-600" data-markers="estacoes" checked>
                                <span class="text-sm">Mostrar Esta\u00e7\u00f5es</span>
                            </label>
                        </div>
                    </div>

                    <!-- Grupo: Barragens -->
                    <div class="border border-gray-200 rounded-lg overflow-hidden">
                        <div class="bg-purple-50 px-3 py-2 font-semibold text-purple-800 text-sm flex items-center justify-between cursor-pointer" onclick="toggleGroup('barragens')">
                            <span><i class="fas fa-dam mr-2"></i>Barragens</span>
                            <i class="fas fa-chevron-down text-purple-600" id="icon-barragens"></i>
                        </div>
                        <div id="group-barragens" class="p-3 space-y-2 bg-white">
                            <label class="flex items-center space-x-2 cursor-pointer hover:bg-purple-50 p-2 rounded">
                                <input type="checkbox" class="marker-toggle form-checkbox text-purple-600" data-markers="barragens" checked>
                                <span class="text-sm">Mostrar Barragens</span>
                            </label>
                        </div>
                    </div>

                    <!-- Grupo: Infraestrutura -->
                    <div class="border border-gray-200 rounded-lg overflow-hidden">
                        <div class="bg-orange-50 px-3 py-2 font-semibold text-orange-800 text-sm flex items-center justify-between cursor-pointer" onclick="toggleGroup('infra')">
                            <span><i class="fas fa-cogs mr-2"></i>Infraestrutura</span>
                            <i class="fas fa-chevron-down text-orange-600" id="icon-infra"></i>
                        </div>
                        <div id="group-infra" class="p-3 space-y-2 bg-white">
                            <label class="flex items-center space-x-2 cursor-pointer hover:bg-orange-50 p-2 rounded">
                                <input type="checkbox" class="layer-toggle form-checkbox text-blue-600" data-layer="rios" data-color="#0066FF">
                                <div class="w-4 h-4 rounded" style="background: #0066FF;"></div>
                                <span class="text-sm">Rios</span>
                            </label>
                            <label class="flex items-center space-x-2 cursor-pointer hover:bg-orange-50 p-2 rounded">
                                <input type="checkbox" class="layer-toggle form-checkbox text-cyan-600" data-layer="albufeiras" data-color="#4169E1">
                                <div class="w-4 h-4 rounded" style="background: #4169E1;"></div>
                                <span class="text-sm">Albufeiras</span>
                            </label>
                        </div>
                    </div>
                </div>

                <!-- Footer do Painel -->
                <div class="border-t border-gray-200 px-4 py-3 bg-gray-50">
                    <div class="text-xs text-gray-600">
                        <i class="fas fa-info-circle mr-1"></i>
                        <span id="layers-count">0</span> camadas ativas
                    </div>
                </div>
            </div>

            <!-- Bot\u00e3o para mostrar painel quando oculto -->
            <button id="show-layers-panel" class="absolute left-4 top-4 bg-white rounded-lg shadow-lg px-3 py-2 z-[999] hidden hover:bg-gray-50 transition-colors">
                <i class="fas fa-layer-group text-blue-600"></i>
            </button>

            <div id="mapa-principal" style="height: 600px; width: 100%;"></div>

            <!-- Legenda removida para interface mais limpa -->

            <!-- Info Panel -->
            <div id="info-panel" class="absolute bottom-4 left-4 bg-white rounded-lg shadow-lg p-4 max-w-sm hidden">
                <div class="flex justify-between items-center mb-2">
                    <h4 class="font-bold text-gray-900">Informações</h4>
                    <button id="close-info" class="text-gray-500 hover:text-gray-700">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                <div id="info-content" class="text-sm text-gray-600">
                    <!-- Conteúdo dinâmico -->
                </div>
            </div>
        </div>
    </div>

    <!-- Ações Rápidas - Design Compacto e Elegante -->
    <div class="bg-white rounded-lg shadow-md border border-gray-100">
        <!-- Header compacto -->
        <div class="px-6 py-4 border-b border-gray-100 bg-gradient-to-r from-gray-50 to-blue-50">
            <div class="flex items-center justify-between">
                <h3 class="text-lg font-semibold text-gray-900 flex items-center">
                    <i class="fas fa-bolt mr-2 text-blue-600"></i>
                    Ações Rápidas
                </h3>
                <span class="text-xs text-gray-500 font-medium">Sistema ARA Norte</span>
            </div>
        </div>

        <!-- Grid compacto das ações -->
        <div class="p-6">
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
                <!-- Nova Bacia -->
                <a href="{{ route('bacias.create') }}" class="group flex items-center p-4 bg-cyan-50 rounded-lg border border-cyan-200 hover:bg-cyan-100 hover:border-cyan-300 hover:shadow-md transition-all duration-200">
                    <div class="w-10 h-10 bg-cyan-500 rounded-lg flex items-center justify-center mr-3 group-hover:scale-110 transition-transform duration-200">
                        <i class="fas fa-plus text-white text-sm"></i>
                    </div>
                    <div class="flex-1 min-w-0">
                        <h4 class="font-semibold text-gray-900 text-sm">Nova Bacia</h4>
                        <p class="text-xs text-cyan-700">Criar bacia hidrográfica</p>
                    </div>
                    <i class="fas fa-arrow-right text-cyan-600 text-sm opacity-0 group-hover:opacity-100 transition-opacity duration-200"></i>
                </a>

                <!-- Nova Barragem -->
                <a href="{{ route('barragens.create') }}" class="group flex items-center p-4 bg-blue-50 rounded-lg border border-blue-200 hover:bg-blue-100 hover:border-blue-300 hover:shadow-md transition-all duration-200">
                    <div class="w-10 h-10 bg-blue-500 rounded-lg flex items-center justify-center mr-3 group-hover:scale-110 transition-transform duration-200">
                        <i class="fas fa-water text-white text-sm"></i>
                    </div>
                    <div class="flex-1 min-w-0">
                        <h4 class="font-semibold text-gray-900 text-sm">Nova Barragem</h4>
                        <p class="text-xs text-blue-700">Registrar infraestrutura</p>
                    </div>
                    <i class="fas fa-arrow-right text-blue-600 text-sm opacity-0 group-hover:opacity-100 transition-opacity duration-200"></i>
                </a>

                <!-- Nova Estação -->
                <a href="{{ route('estacoes.create') }}" class="group flex items-center p-4 bg-green-50 rounded-lg border border-green-200 hover:bg-green-100 hover:border-green-300 hover:shadow-md transition-all duration-200">
                    <div class="w-10 h-10 bg-green-500 rounded-lg flex items-center justify-center mr-3 group-hover:scale-110 transition-transform duration-200">
                        <i class="fas fa-satellite-dish text-white text-sm"></i>
                    </div>
                    <div class="flex-1 min-w-0">
                        <h4 class="font-semibold text-gray-900 text-sm">Nova Estação</h4>
                        <p class="text-xs text-green-700">Configurar monitoramento</p>
                    </div>
                    <i class="fas fa-arrow-right text-green-600 text-sm opacity-0 group-hover:opacity-100 transition-opacity duration-200"></i>
                </a>

                <!-- Alertas -->
                <a href="{{ route('alertas.index') }}" class="group flex items-center p-4 {{ ($stats['alertas_ativos'] ?? 0) > 0 ? 'bg-red-50 border-red-200 hover:bg-red-100 hover:border-red-300' : 'bg-gray-50 border-gray-200 hover:bg-gray-100 hover:border-gray-300' }} rounded-lg border hover:shadow-md transition-all duration-200">
                    <div class="w-10 h-10 {{ ($stats['alertas_ativos'] ?? 0) > 0 ? 'bg-red-500' : 'bg-gray-500' }} rounded-lg flex items-center justify-center mr-3 group-hover:scale-110 transition-transform duration-200">
                        <i class="fas fa-exclamation-triangle text-white text-sm {{ ($stats['alertas_ativos'] ?? 0) > 0 ? 'animate-pulse' : '' }}"></i>
                    </div>
                    <div class="flex-1 min-w-0">
                        <div class="flex items-center space-x-2">
                            <h4 class="font-semibold {{ ($stats['alertas_ativos'] ?? 0) > 0 ? 'text-red-900' : 'text-gray-900' }} text-sm">
                                {{ ($stats['alertas_ativos'] ?? 0) > 0 ? 'Alertas Críticos' : 'Alertas' }}
                            </h4>
                            @if(($stats['alertas_ativos'] ?? 0) > 0)
                                <span class="bg-red-500 text-white text-xs font-bold px-1.5 py-0.5 rounded-full">{{ $stats['alertas_ativos'] }}</span>
                            @endif
                        </div>
                        <p class="text-xs {{ ($stats['alertas_ativos'] ?? 0) > 0 ? 'text-red-700' : 'text-gray-700' }}">
                            {{ ($stats['alertas_ativos'] ?? 0) > 0 ? 'Atenção necessária' : 'Centro de monitoramento' }}
                        </p>
                    </div>
                    <i class="fas fa-arrow-right {{ ($stats['alertas_ativos'] ?? 0) > 0 ? 'text-red-600' : 'text-gray-600' }} text-sm opacity-0 group-hover:opacity-100 transition-opacity duration-200"></i>
                </a>
            </div>

            <!-- Ações secundárias compactas -->
            <div class="pt-4 border-t border-gray-100">
                <div class="flex flex-wrap gap-2 justify-center">
                    <a href="{{ route('boletins.index') }}" class="inline-flex items-center px-3 py-2 bg-purple-100 hover:bg-purple-200 text-purple-800 rounded-md text-xs font-medium transition-colors duration-200 group">
                        <i class="fas fa-file-alt mr-1.5 group-hover:scale-110 transition-transform duration-200"></i>
                        Boletins
                    </a>
                    <a href="{{ route('charts.index') }}" class="inline-flex items-center px-3 py-2 bg-orange-100 hover:bg-orange-200 text-orange-800 rounded-md text-xs font-medium transition-colors duration-200 group">
                        <i class="fas fa-chart-bar mr-1.5 group-hover:scale-110 transition-transform duration-200"></i>
                        Gráficos
                    </a>
                    <a href="{{ route('configuracoes.index') }}" class="inline-flex items-center px-3 py-2 bg-indigo-100 hover:bg-indigo-200 text-indigo-800 rounded-md text-xs font-medium transition-colors duration-200 group">
                        <i class="fas fa-cog mr-1.5 group-hover:scale-110 transition-transform duration-200"></i>
                        Configurações
                    </a>
                    <a href="{{ route('importacao.index') }}" class="inline-flex items-center px-3 py-2 bg-teal-100 hover:bg-teal-200 text-teal-800 rounded-md text-xs font-medium transition-colors duration-200 group">
                        <i class="fas fa-upload mr-1.5 group-hover:scale-110 transition-transform duration-200"></i>
                        Importar
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Leaflet Core -->
<link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>

<!-- Leaflet Plugins for GIS -->
<script src="https://unpkg.com/leaflet-draw@1.0.4/dist/leaflet.draw.js"></script>
<link rel="stylesheet" href="https://unpkg.com/leaflet-draw@1.0.4/dist/leaflet.draw.css" />

<!-- Esri Leaflet for ArcGIS shapes -->
<script src="https://unpkg.com/esri-leaflet@3.0.10/dist/esri-leaflet.js"></script>

<!-- Leaflet measure plugin -->
<script src="https://cdn.jsdelivr.net/npm/leaflet-measure@3.1.0/dist/leaflet-measure.min.js"></script>
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/leaflet-measure@3.1.0/dist/leaflet-measure.css" />

<!-- Shapefile.js para carregar arquivos .shp diretamente -->
<script src="https://unpkg.com/shapefile@0.6.6/dist/shapefile.min.js"></script>

<!-- Leaflet Coordinates Display -->
<link rel="stylesheet" href="https://unpkg.com/leaflet-control-coordinates@1.0.0/dist/Control.Coordinates.css" />
<script src="https://unpkg.com/leaflet-control-coordinates@1.0.0/dist/Control.Coordinates.min.js"></script>

<!-- Leaflet Easy Button (para bot\u00f5es customizados) -->
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/leaflet-easybutton@2.4.0/src/easy-button.css">
<script src="https://cdn.jsdelivr.net/npm/leaflet-easybutton@2.4.0/src/easy-button.js"></script>

<script>
// Funções globais para controle do painel
function toggleGroup(groupId) {
    const group = document.getElementById(`group-${groupId}`);
    const icon = document.getElementById(`icon-${groupId}`);

    if (group.style.display === 'none') {
        group.style.display = 'block';
        icon.classList.remove('fa-chevron-right');
        icon.classList.add('fa-chevron-down');
    } else {
        group.style.display = 'none';
        icon.classList.remove('fa-chevron-down');
        icon.classList.add('fa-chevron-right');
    }
}

window.addEventListener('load', async function() {
    try {
        console.log('🚀 Iniciando Visor GIS Profissional Estilo SIRHAN...');

        // Verificar se o container do mapa existe
        const mapaContainer = document.getElementById('mapa-principal');
        if (!mapaContainer) {
            throw new Error('Container do mapa não encontrado');
        }

        // Aguardar um pouco para garantir que o DOM está pronto
        await new Promise(resolve => setTimeout(resolve, 200));

        // Controle do painel lateral
        const layersPanel = document.getElementById('layers-panel');
        const togglePanelBtn = document.getElementById('toggle-layers-panel');
        const showPanelBtn = document.getElementById('show-layers-panel');

        togglePanelBtn.addEventListener('click', function() {
            layersPanel.classList.add('hidden');
            showPanelBtn.classList.remove('hidden');
        });

        showPanelBtn.addEventListener('click', function() {
            layersPanel.classList.remove('hidden');
            showPanelBtn.classList.add('hidden');
        });

        // Inicializar mapa com coordenadas para Norte de Moçambique
        var mapa = L.map('mapa-principal', {
            center: [-14.5, 38.5],
            zoom: 7,
            zoomControl: true,
            attributionControl: true
        });

        // Camadas base
        var topoLayer = L.tileLayer('https://{s}.tile.opentopomap.org/{z}/{x}/{y}.png', {
            attribution: '© OpenStreetMap contributors, SRTM | Ideal para sistemas hidrológicos',
            maxZoom: 15
        });

        var satelliteLayer = L.tileLayer('https://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}', {
            attribution: '© Esri | Imagens de satélite',
            maxZoom: 18
        });

        var streetLayer = L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            attribution: '© OpenStreetMap contributors'
        });

        // Adicionar camada topográfica por padrão
        topoLayer.addTo(mapa);

        // 🎨 FERRAMENTAS PROFISSIONAIS ESTILO SIRHAN

        // 1. Controle de Coordenadas do Mouse (se disponível)
        try {
            if (typeof L.control.coordinates !== 'undefined') {
                L.control.coordinates({
                    position: 'bottomright',
                    decimals: 5,
                    decimalSeperator: '.',
                    labelTemplateLat: 'Lat: {y}',
                    labelTemplateLng: 'Lng: {x}',
                    enableUserInput: true,
                    useDMS: false,
                    useLatLngOrder: true
                }).addTo(mapa);
                console.log('✅ Coordenadas ativadas');
            }
        } catch(e) {
            console.warn('⚠️ Controle de coordenadas não disponível:', e.message);
        }

        // 3. Controle de Escala (sempre disponível)
        L.control.scale({
            position: 'bottomleft',
            metric: true,
            imperial: false,
            maxWidth: 200
        }).addTo(mapa);
        console.log('✅ Escala ativada');

        // 4. Ferramentas de Medição (se disponível)
        try {
            if (typeof L.Control.Measure !== 'undefined') {
                var measureControl = new L.Control.Measure({
                    position: 'topleft',
                    primaryLengthUnit: 'kilometers',
                    secondaryLengthUnit: 'meters',
                    primaryAreaUnit: 'hectares',
                    secondaryAreaUnit: 'sqmeters',
                    activeColor: '#3388ff',
                    completedColor: '#38a3ff'
                });
                measureControl.addTo(mapa);
                console.log('✅ Ferramentas de medição ativadas');
            }
        } catch(e) {
            console.warn('⚠️ Ferramentas de medição não disponíveis:', e.message);
        }

        // 5. Botões customizados (se disponível)
        try {
            if (typeof L.easyButton !== 'undefined') {
                // Botão para voltar à área ARA Norte
                L.easyButton('<i class="fas fa-home"></i>', function(btn, map){
                    map.setView([-14.5, 38.5], 7);
                }, 'Voltar à visão inicial').addTo(mapa);

                // Botão para localização atual
                L.easyButton('<i class="fas fa-crosshairs"></i>', function(btn, map){
                    map.locate({setView: true, maxZoom: 12});
                }, 'Ir para minha localização').addTo(mapa);

                console.log('✅ Botões customizados ativados');
            }
        } catch(e) {
            console.warn('⚠️ Botões customizados não disponíveis:', e.message);
        }

        //7. Atualizar zoom display
        mapa.on('zoomend', function() {
            const zoomDisplay = document.getElementById('current-zoom');
            if (zoomDisplay) {
                zoomDisplay.textContent = mapa.getZoom();
            }
        });

        // Controle de camadas base
        var baseLayers = {
            "🗻 Topográfico (Recomendado)": topoLayer,
            "🛰️ Satélite": satelliteLayer,
            "🗺️ Mapa Padrão": streetLayer
        };

        // Controle de troca de camadas base via painel lateral
        document.querySelectorAll('input[name="base-layer"]').forEach(radio => {
            radio.addEventListener('change', function() {
                // Remover todas as camadas base
                mapa.removeLayer(topoLayer);
                mapa.removeLayer(satelliteLayer);
                mapa.removeLayer(streetLayer);

                // Adicionar camada selecionada
                if (this.value === 'topo') {
                    topoLayer.addTo(mapa);
                } else if (this.value === 'satellite') {
                    satelliteLayer.addTo(mapa);
                } else if (this.value === 'standard') {
                    streetLayer.addTo(mapa);
                }
            });
        });

        // Grupos de marcadores por categoria
        var markersBarragens = L.layerGroup();
        var markersEstacoes = L.layerGroup();
        var markersBacias = L.layerGroup();

        // Objeto para armazenar camadas de shapes carregadas
        var allShapeLayers = {};

        // Contadores para a legenda
        var counts = {
            barragens: 0,
            estacoes: 0,
            bacias: 0
        };

        // Dados do backend
        var dadosBarragens = @json($barragens_mapa ?? []);
        var dadosEstacoes = @json($estacoes_mapa ?? []);
        var dadosBacias = @json($bacias_mapa ?? []);

        // Função para criar ícones seguindo padrão internacional WMO (World Meteorological Organization)
        // Referência: WMO Guide to Hydrological Practices (WMO-No. 168)
        function criarIcone(tipo, cor) {
            let html, iconSize, iconAnchor;

            if (tipo === 'barragem') {
                // Barragem/Represa - Triângulo sólido preto apontando para baixo (ISO 19115/WMO padrão)
                html = '<div style="width: 0; height: 0; border-left: 12px solid transparent; border-right: 12px solid transparent; border-bottom: 20px solid #000000; filter: drop-shadow(0 2px 4px rgba(0,0,0,0.5));"></div>';
                iconSize = [24, 24];
                iconAnchor = [12, 20];
            } else if (tipo === 'estacao') {
                // Estação - Símbolo diferenciado por tipo
                if (cor === '#16a34a') {
                    // Estação Hidrométrica - Círculo azul sólido (WMO padrão para medição de nível)
                    html = '<div style="background-color: #1e40af; width: 18px; height: 18px; border-radius: 50%; border: 3px solid white; box-shadow: 0 2px 6px rgba(0,0,0,0.4);"></div>';
                    iconSize = [24, 24];
                    iconAnchor = [12, 12];
                } else if (cor === '#0891b2') {
                    // Estação Pluviométrica - Triângulo verde apontando para cima (WMO padrão para precipitação)
                    html = '<div style="width: 0; height: 0; border-left: 12px solid transparent; border-right: 12px solid transparent; border-top: 20px solid #16a34a; filter: drop-shadow(0 2px 4px rgba(0,0,0,0.5));"></div>';
                    iconSize = [24, 24];
                    iconAnchor = [12, 4];
                } else {
                    // Outras estações - Círculo padrão
                    html = `<div style="background-color: ${cor}; width: 18px; height: 18px; border-radius: 50%; border: 3px solid white; box-shadow: 0 2px 6px rgba(0,0,0,0.4);"></div>`;
                    iconSize = [24, 24];
                    iconAnchor = [12, 12];
                }
            } else {
                // Bacias e outros - Quadrado
                html = `<div style="background-color: ${cor}; width: 14px; height: 14px; border: 2px solid white; box-shadow: 0 2px 4px rgba(0,0,0,0.3);"></div>`;
                iconSize = [18, 18];
                iconAnchor = [9, 9];
            }

            return L.divIcon({
                html: html,
                className: 'custom-div-icon',
                iconSize: iconSize,
                iconAnchor: iconAnchor
            });
        }

        // Adicionar barragens
        dadosBarragens.forEach(function(barragem) {
            if (barragem.latitude && barragem.longitude) {
                var marker = L.marker([barragem.latitude, barragem.longitude], {
                    icon: criarIcone('barragem', '#2563eb')
                }).bindPopup(`
                    <div class="p-3">
                        <h4 class="font-bold text-blue-800 mb-2">${barragem.nome}</h4>
                        <p><strong>Código:</strong> ${barragem.codigo}</p>
                        <p><strong>Tipo:</strong> Barragem</p>
                        ${barragem.altura ? `<p><strong>Altura:</strong> ${barragem.altura}m</p>` : ''}
                        ${barragem.cota_atual ? `<p><strong>Cota Atual:</strong> ${barragem.cota_atual}m</p>` : ''}
                        ${barragem.percentagem_enchimento ? `<p><strong>Enchimento:</strong> ${barragem.percentagem_enchimento}%</p>` : ''}
                        <div class="mt-3 pt-2 border-t border-gray-200">
                            <a href="/barragens/${barragem.id}" class="inline-flex items-center px-3 py-2 bg-blue-600 hover:bg-blue-700 text-white text-sm font-medium rounded-lg transition-colors duration-200">
                                <i class="fas fa-arrow-right mr-2"></i>Ver Detalhes
                            </a>
                        </div>
                    </div>
                `);
                markersBarragens.addLayer(marker);
                counts.barragens++;
            }
        });

        // Adicionar estações
        dadosEstacoes.forEach(function(estacao) {
            if (estacao.latitude && estacao.longitude) {
                var cor = estacao.tipo === 'hidrometrica' ? '#16a34a' :
                         estacao.tipo === 'pluviometrica' ? '#0891b2' : '#7c3aed';

                var marker = L.marker([estacao.latitude, estacao.longitude], {
                    icon: criarIcone('estacao', cor)
                }).bindPopup(`
                    <div class="p-3">
                        <h4 class="font-bold text-green-800 mb-2">${estacao.nome}</h4>
                        <p><strong>Código:</strong> ${estacao.codigo}</p>
                        <p><strong>Tipo:</strong> ${estacao.tipo}</p>
                        <p><strong>Estado:</strong> ${estacao.estado}</p>
                        ${estacao.provincia ? `<p><strong>Província:</strong> ${estacao.provincia}</p>` : ''}
                        <div class="mt-3 pt-2 border-t border-gray-200">
                            <a href="/estacoes/${estacao.id}" class="inline-flex items-center px-3 py-2 bg-green-600 hover:bg-green-700 text-white text-sm font-medium rounded-lg transition-colors duration-200">
                                <i class="fas fa-arrow-right mr-2"></i>Ver Detalhes
                            </a>
                        </div>
                    </div>
                `);
                markersEstacoes.addLayer(marker);
                counts.estacoes++;
            }
        });

        // Adicionar bacias
        dadosBacias.forEach(function(bacia) {
            if (bacia.latitude && bacia.longitude) {
                var marker = L.marker([bacia.latitude, bacia.longitude], {
                    icon: criarIcone('bacia', '#0891b2')
                }).bindPopup(`
                    <div class="p-3">
                        <h4 class="font-bold text-cyan-800 mb-2">${bacia.nome}</h4>
                        <p><strong>Código:</strong> ${bacia.codigo}</p>
                        <p><strong>Tipo:</strong> Bacia Hidrográfica</p>
                        ${bacia.area_km2 ? `<p><strong>Área:</strong> ${bacia.area_km2} km²</p>` : ''}
                        ${bacia.provincias_abrangidas ? `<p><strong>Províncias:</strong> ${bacia.provincias_abrangidas}</p>` : ''}
                        <div class="mt-3 pt-2 border-t border-gray-200">
                            <a href="/bacias/${bacia.id}" class="inline-flex items-center px-3 py-2 bg-cyan-600 hover:bg-cyan-700 text-white text-sm font-medium rounded-lg transition-colors duration-200">
                                <i class="fas fa-arrow-right mr-2"></i>Ver Detalhes
                            </a>
                        </div>
                    </div>
                `);
                markersBacias.addLayer(marker);
                counts.bacias++;
            }
        });

        // Adicionar todas as camadas ao mapa
        markersBarragens.addTo(mapa);
        markersEstacoes.addTo(mapa);
        markersBacias.addTo(mapa);

        // 🎯 CONECTAR PAINEL LATERAL COM CAMADAS DO MAPA

        // Controle de marcadores (estações, barragens, bacias)
        document.querySelectorAll('.marker-toggle').forEach(checkbox => {
            checkbox.addEventListener('change', function() {
                const markerType = this.getAttribute('data-markers');

                if (markerType === 'estacoes') {
                    if (this.checked) {
                        markersEstacoes.addTo(mapa);
                    } else {
                        mapa.removeLayer(markersEstacoes);
                    }
                } else if (markerType === 'barragens') {
                    if (this.checked) {
                        markersBarragens.addTo(mapa);
                    } else {
                        mapa.removeLayer(markersBarragens);
                    }
                } else if (markerType === 'bacias') {
                    if (this.checked) {
                        markersBacias.addTo(mapa);
                    } else {
                        mapa.removeLayer(markersBacias);
                    }
                }

                updateLayersCount();
            });
        });

        // Controle de camadas shapes (bacias, rios, etc)
        document.querySelectorAll('.layer-toggle').forEach(checkbox => {
            checkbox.addEventListener('change', function() {
                const layerKey = this.getAttribute('data-layer');
                const layerColor = this.getAttribute('data-color');

                const shapeFiles = {
                    'bacia_rovuma': { file: 'Bacia_Rovuma', color: '#E74C3C', name: 'Bacia do Rovuma' },
                    'bacia_messalo': { file: 'Bacia_Messalo', color: '#2ECC71', name: 'Bacia do Messalo' },
                    'bacia_licungo': { file: 'Bacia_Licungo', color: '#3498DB', name: 'Bacia do Lúrio' },
                    'bacia_monapo': { file: 'Bacia_Monapo', color: '#F39C12', name: 'Bacia do Monapo' },
                    'bacia_meluli': { file: 'Bacia_Meluli', color: '#9B59B6', name: 'Bacia do Meluli' },
                    'bacia_mecuburi': { file: 'Bacia_Mecuburi', color: '#1ABC9C', name: 'Bacia do Mecuburi' },
                    'megarruma': { file: 'Megarruma', color: '#E91E63', name: 'Bacia Megarruma' },
                    'rios': { file: 'Rios', color: '#0066FF', name: 'Rios da ARA Norte' },
                    'albufeiras': { file: 'Albufeiras', color: '#4169E1', name: 'Albufeiras' }
                };

                if (this.checked) {
                    // Carregar camada se não existir
                    if (!allShapeLayers[layerKey] && shapeFiles[layerKey]) {
                        const config = shapeFiles[layerKey];
                        console.log(`🔄 Carregando ${config.name}...`);

                        // Mostrar loading no painel
                        this.parentElement.classList.add('opacity-50');

                        carregarShapeReal(config.file, layerKey, config.color, config.name)
                            .then(layer => {
                                if (layer) {
                                    layer.addTo(mapa);
                                    console.log(`✅ ${config.name} carregada!`);
                                }
                                this.parentElement.classList.remove('opacity-50');
                                updateLayersCount();
                            })
                            .catch(error => {
                                console.error(`❌ Erro ao carregar ${config.name}:`, error);
                                this.checked = false;
                                this.parentElement.classList.remove('opacity-50');
                                alert(`Erro ao carregar ${config.name}`);
                            });
                    } else if (allShapeLayers[layerKey]) {
                        allShapeLayers[layerKey].addTo(mapa);
                        updateLayersCount();
                    }
                } else {
                    // Remover camada
                    if (allShapeLayers[layerKey]) {
                        mapa.removeLayer(allShapeLayers[layerKey]);
                        updateLayersCount();
                    }
                }
            });
        });

        // Função para atualizar contagem de camadas ativas
        function updateLayersCount() {
            let count = 0;

            // Contar marcadores ativos
            if (mapa.hasLayer(markersBarragens)) count++;
            if (mapa.hasLayer(markersEstacoes)) count++;
            if (mapa.hasLayer(markersBacias)) count++;

            // Contar shapes ativos
            Object.values(allShapeLayers).forEach(layer => {
                if (layer && mapa.hasLayer(layer)) count++;
            });

            const layersCountEl = document.getElementById('layers-count');
            if (layersCountEl) {
                layersCountEl.textContent = count;
            }
        }

        // Atualizar contagem inicial
        updateLayersCount();

        // Overlays para controle de camadas (marcadores existentes)
        var overlays = {
            "🏗️ Barragens": markersBarragens,
            "📊 Estações": markersEstacoes,
            "🌊 Bacias": markersBacias
        };

        // Controle de camadas básico (será atualizado depois com shapes)
        var layerControl = L.control.layers(baseLayers, overlays);
        layerControl.addTo(mapa);



        // Atualizar contadores na legenda (com verificação)
        const countBarragens = document.getElementById('count-barragens');
        if (countBarragens) countBarragens.textContent = counts.barragens;

        const countEstacoes = document.getElementById('count-estacoes');
        if (countEstacoes) countEstacoes.textContent = counts.estacoes;

        const countBacias = document.getElementById('count-bacias');
        if (countBacias) countBacias.textContent = counts.bacias;

        // Atualização de coordenadas do mouse (com verificação)
        mapa.on('mousemove', function(e) {
            var lat = e.latlng.lat.toFixed(4);
            var lng = e.latlng.lng.toFixed(4);
            const coordsElement = document.getElementById('mouse-coords');
            if (coordsElement) {
                coordsElement.textContent = `${lat}, ${lng}`;
            }
        });

        // Atualização do zoom (com verificação)
        mapa.on('zoomend', function() {
            const zoomElement = document.getElementById('current-zoom');
            if (zoomElement) {
                zoomElement.textContent = mapa.getZoom();
            }
        });

        // Plugins avançados removidos para interface mais limpa
        // Foco no carregamento de shapes GIS profissionais

        // Implementar filtros
        function aplicarFiltros() {
            var filtroBacia = document.getElementById('filtro-bacia').value;
            var filtroProvincia = document.getElementById('filtro-provincia').value;
            var filtroTipo = document.getElementById('filtro-tipo').value;

            // Limpar marcadores atuais
            markersBarragens.clearLayers();
            markersEstacoes.clearLayers();
            markersBacias.clearLayers();

            // Recontadores
            counts = { barragens: 0, estacoes: 0, bacias: 0 };

            // Filtrar e re-adicionar barragens
            dadosBarragens.forEach(function(barragem) {
                var mostrar = true;

                if (filtroTipo && filtroTipo !== 'barragem') mostrar = false;

                if (mostrar && barragem.latitude && barragem.longitude) {
                    var marker = L.marker([barragem.latitude, barragem.longitude], {
                        icon: criarIcone('barragem', '#2563eb')
                    }).bindPopup(`
                        <div class="p-2">
                            <h4 class="font-bold text-blue-800">${barragem.nome}</h4>
                            <p><strong>Código:</strong> ${barragem.codigo}</p>
                            <p><strong>Tipo:</strong> Barragem</p>
                        </div>
                    `);
                    markersBarragens.addLayer(marker);
                    counts.barragens++;
                }
            });

            // Filtrar e re-adicionar estações
            dadosEstacoes.forEach(function(estacao) {
                var mostrar = true;

                if (filtroProvincia && estacao.provincia !== filtroProvincia) mostrar = false;
                if (filtroTipo && !['hidrometrica', 'pluviometrica', 'meteorologica'].includes(filtroTipo) && filtroTipo !== estacao.tipo) mostrar = false;

                if (mostrar && estacao.latitude && estacao.longitude) {
                    var cor = estacao.tipo === 'hidrometrica' ? '#16a34a' :
                             estacao.tipo === 'pluviometrica' ? '#0891b2' : '#7c3aed';

                    var marker = L.marker([estacao.latitude, estacao.longitude], {
                        icon: criarIcone('estacao', cor)
                    }).bindPopup(`
                        <div class="p-2">
                            <h4 class="font-bold text-green-800">${estacao.nome}</h4>
                            <p><strong>Código:</strong> ${estacao.codigo}</p>
                            <p><strong>Tipo:</strong> ${estacao.tipo}</p>
                            <p><strong>Província:</strong> ${estacao.provincia}</p>
                        </div>
                    `);
                    markersEstacoes.addLayer(marker);
                    counts.estacoes++;
                }
            });

            // Filtrar e re-adicionar bacias
            dadosBacias.forEach(function(bacia) {
                var mostrar = true;

                if (filtroBacia && bacia.id != filtroBacia) mostrar = false;
                if (filtroTipo && filtroTipo !== 'bacia') mostrar = false;

                if (mostrar && bacia.latitude && bacia.longitude) {
                    var marker = L.marker([bacia.latitude, bacia.longitude], {
                        icon: criarIcone('bacia', '#0891b2')
                    }).bindPopup(`
                        <div class="p-2">
                            <h4 class="font-bold text-cyan-800">${bacia.nome}</h4>
                            <p><strong>Código:</strong> ${bacia.codigo}</p>
                            <p><strong>Tipo:</strong> Bacia Hidrográfica</p>
                        </div>
                    `);
                    markersBacias.addLayer(marker);
                    counts.bacias++;
                }
            });

            // Atualizar contadores (com verificação)
            const countBarragens2 = document.getElementById('count-barragens');
            if (countBarragens2) countBarragens2.textContent = counts.barragens;

            const countEstacoes2 = document.getElementById('count-estacoes');
            if (countEstacoes2) countEstacoes2.textContent = counts.estacoes;

            const countBacias2 = document.getElementById('count-bacias');
            if (countBacias2) countBacias2.textContent = counts.bacias;
        }

        // Event listeners para filtros (removidos filtros antigos não utilizados)

        // Nota: allShapeLayers já foi declarado no início como objeto vazio {}
        // Será preenchido dinamicamente conforme as camadas são carregadas

        // Função para carregar qualquer shape (como programa GIS)
        function carregarShape(shapeName, layerKey, cor = '#2563eb', autoAdd = true) {
            console.log(`🔄 Carregando: ${shapeName}...`);

            if (allShapeLayers[layerKey]) {
                mapa.removeLayer(allShapeLayers[layerKey]);
            }

            // Carregar GeoJSON
            const geojsonUrl = `/geojson/${shapeName}.geojson`;

            return fetch(geojsonUrl)
                .then(response => {
                    if (!response.ok) {
                        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                    }
                    return response.json();
                })
                .then(geojson => {
                    console.log(`✅ ${shapeName}: ${geojson.features.length} features`);

                    // Criar layer
                    allShapeLayers[layerKey] = L.geoJSON(geojson, {
                        style: function(feature) {
                            const geometry = feature.geometry.type;
                            return {
                                color: cor,
                                weight: geometry.includes('Line') ? 3 : 2,
                                opacity: 0.8,
                                fillOpacity: geometry.includes('Polygon') ? 0.2 : 0
                            };
                        },
                        pointToLayer: function(feature, latlng) {
                            return L.circleMarker(latlng, {
                                radius: 6,
                                fillColor: cor,
                                color: cor,
                                weight: 2,
                                opacity: 1,
                                fillOpacity: 0.8
                            });
                        },
                        onEachFeature: function(feature, layer) {
                            let popupContent = `<div class="p-2">
                                <h4 class="font-bold" style="color: ${cor};">🗺️ ${feature.properties.name || shapeName}</h4>`;

                            // Mostrar todas as propriedades
                            Object.keys(feature.properties || {}).forEach(key => {
                                if (feature.properties[key] && key !== 'name') {
                                    popupContent += `<p><strong>${key}:</strong> ${feature.properties[key]}</p>`;
                                }
                            });

                            popupContent += `<p class="text-xs mt-1"><em>📊 ARA Norte (${geojson.features.length} features)</em></p></div>`;
                            layer.bindPopup(popupContent);
                        }
                    });

                    // Adicionar ao mapa se solicitado
                    if (autoAdd) {
                        allShapeLayers[layerKey].addTo(mapa);
                    }

                    // Adicionar ao controle de camadas se existir
                    if (typeof layerControl !== 'undefined') {
                        layerControl.addOverlay(allShapeLayers[layerKey], `📊 ${shapeName}`);
                    }

                    console.log(`🎉 ${shapeName} carregado!`);
                    return allShapeLayers[layerKey];
                })
                .catch(error => {
                    console.error(`❌ Erro ${shapeName}:`, error);
                    return null;
                });
        }

        // Função para carregar shapes REAIS usando shapefile.js
        function carregarShapeReal(shapeBaseName, layerKey, cor, nome) {
            console.log(`🔄 Carregando shape REAL: ${shapeBaseName}...`);

            if (allShapeLayers[layerKey]) {
                mapa.removeLayer(allShapeLayers[layerKey]);
                allShapeLayers[layerKey] = null;
            }

            // URLs para os arquivos reais
            const shpUrl = `/shapes/${shapeBaseName}.shp`;
            const dbfUrl = `/shapes/${shapeBaseName}.dbf`;

            // Verificar se shapefile.js está disponível
            if (typeof shapefile === 'undefined') {
                console.error('❌ shapefile.js não carregado');
                return Promise.resolve(null);
            }

            return shapefile.open(shpUrl, dbfUrl)
                .then(source => {
                    console.log(`📁 Arquivo ${shapeBaseName} aberto, lendo geometrias reais...`);

                    const features = [];
                    let featureCount = 0;

                    return source.read().then(function collect(result) {
                        if (result.done) {
                            console.log(`✅ ${shapeBaseName}: ${features.length} geometrias REAIS carregadas`);

                            if (features.length === 0) {
                                console.warn(`⚠️ ${shapeBaseName}: Nenhuma feature encontrada`);
                                return null;
                            }

                            // Criar GeoJSON com geometrias reais
                            const geojson = {
                                type: 'FeatureCollection',
                                features: features
                            };

                            // Debug: Mostrar bounds da bacia
                            if (features.length > 0 && features[0].geometry) {
                                console.log(`🗺️ ${shapeBaseName} - Tipo geometria:`, features[0].geometry.type);
                                console.log(`📍 ${shapeBaseName} - Primeira coordenada:`,
                                    features[0].geometry.coordinates[0] ?
                                    features[0].geometry.coordinates[0][0] :
                                    features[0].geometry.coordinates);
                            }

                            // Criar camada GeoJSON com estilo apropriado
                            allShapeLayers[layerKey] = L.geoJSON(geojson, {
                                style: function(feature) {
                                    const geomType = feature.geometry.type;

                                    if (geomType.includes('Line')) {
                                        // Rios e divisores
                                        return {
                                            color: cor,
                                            weight: 2,
                                            opacity: 0.8
                                        };
                                    } else if (geomType.includes('Polygon')) {
                                        // Bacias e áreas
                                        return {
                                            color: cor,
                                            weight: 2,
                                            opacity: 0.8,
                                            fillOpacity: 0.15
                                        };
                                    } else {
                                        // Pontos (barragens, etc.)
                                        return {
                                            color: cor,
                                            weight: 2,
                                            opacity: 1,
                                            fillOpacity: 0.8
                                        };
                                    }
                                },
                                pointToLayer: function(feature, latlng) {
                                    return L.circleMarker(latlng, {
                                        radius: 6,
                                        fillColor: cor,
                                        color: '#fff',
                                        weight: 2,
                                        opacity: 1,
                                        fillOpacity: 0.9
                                    });
                                },
                                onEachFeature: function(feature, layer) {
                                    // Popup com dados REAIS do shape
                                    let popupContent = `<div class="p-3">
                                        <h4 class="font-bold text-lg mb-2" style="color: ${cor};">🗺️ ${nome}</h4>`;

                                    // Mostrar TODAS as propriedades do shape real
                                    if (feature.properties && Object.keys(feature.properties).length > 0) {
                                        popupContent += '<div class="space-y-1 text-sm">';
                                        Object.keys(feature.properties).forEach(key => {
                                            const value = feature.properties[key];
                                            if (value !== null && value !== undefined && value !== '') {
                                                popupContent += `<div><strong>${key}:</strong> ${value}</div>`;
                                            }
                                        });
                                        popupContent += '</div>';
                                    }

                                    popupContent += `<div class="mt-2 text-xs text-gray-600 border-t pt-2">
                                        <div>📊 Arquivo: ${shapeBaseName}.shp</div>
                                        <div>🎯 Geometria: ${feature.geometry.type}</div>
                                        <div>📈 Total features: ${features.length}</div>
                                        <div>🏛️ Fonte: ARA Norte</div>
                                    </div></div>`;

                                    layer.bindPopup(popupContent, {
                                        maxWidth: 400,
                                        className: 'custom-popup'
                                    });
                                }
                            }).addTo(mapa);

                            // Adicionar ao controle de camadas com categorização
                            if (typeof layerControl !== 'undefined') {
                                // Categorizar por tipo
                                let categoria = '';
                                let icon = '';

                                if (shapeBaseName.toLowerCase().includes('rio')) {
                                    categoria = '🌊 Rios (Shapes)';
                                    icon = '🌊';
                                } else if (shapeBaseName.toLowerCase().includes('bacia')) {
                                    categoria = '💧 Bacias Hidrográficas';
                                    icon = '💧';
                                } else if (shapeBaseName.toLowerCase().includes('ara_norte') || shapeBaseName.toLowerCase().includes('ara norte')) {
                                    categoria = '🏛️ Área ARA Norte';
                                    icon = '🏛️';
                                } else if (shapeBaseName.toLowerCase().includes('dgb') || shapeBaseName.toLowerCase().includes('divisor')) {
                                    categoria = '🌄 Divisores Simulados';
                                    icon = '🌄';
                                } else if (shapeBaseName.toLowerCase().includes('barrage') || shapeBaseName.toLowerCase().includes('albufeira')) {
                                    categoria = '🏗️ Infraestrutura Hídrica';
                                    icon = '🏗️';
                                } else {
                                    categoria = '📊 Outros Shapes';
                                    icon = '📊';
                                }

                                layerControl.addOverlay(allShapeLayers[layerKey], `${icon} ${nome}`);
                            }

                            console.log(`🎉 ${nome} adicionado ao mapa com ${features.length} geometrias reais!`);
                            return allShapeLayers[layerKey];

                        } else {
                            // Adicionar feature com geometria real
                            if (result.value && result.value.geometry) {
                                features.push(result.value);
                                featureCount++;

                                // Log de progresso para shapes grandes
                                if (featureCount % 100 === 0) {
                                    console.log(`   📊 ${featureCount} features processadas...`);
                                }
                            }
                            return source.read().then(collect);
                        }
                    });
                })
                .catch(error => {
                    console.error(`❌ Erro ao carregar ${shapeBaseName}:`, error);
                    return null;
                });
        }

        // Função para carregar TODOS os shapes REAIS automaticamente
        function carregarTodosOsShapesReais() {
            console.log('🗺️ CARREGANDO TODOS OS SHAPES REAIS DA ARA NORTE...');
            console.log('📁 Usando geometrias originais dos arquivos .shp');

            // Configuração dos shapes REAIS disponíveis - Cores distintas para cada bacia
            const shapesReais = [
                { file: 'Rios', key: 'rios', color: '#0066FF', name: 'Rios da ARA Norte' },
                { file: 'ARA_Norte_IP', key: 'ara_norte_ip', color: '#FF7F00', name: 'Área ARA Norte' },
                { file: 'Bacia_Rovuma', key: 'bacia_rovuma', color: '#E74C3C', name: 'Bacia do Rovuma' },        // Vermelho forte
                { file: 'Bacia_Messalo', key: 'bacia_messalo', color: '#2ECC71', name: 'Bacia do Messalo' },      // Verde forte
                { file: 'Bacia_Licungo', key: 'bacia_licungo', color: '#3498DB', name: 'Bacia do Lúrio' },        // Azul forte
                { file: 'Bacia_Monapo', key: 'bacia_monapo', color: '#F39C12', name: 'Bacia do Monapo' },         // Laranja forte
                { file: 'Bacia_Meluli', key: 'bacia_meluli', color: '#9B59B6', name: 'Bacia do Meluli' },         // Roxo forte
                { file: 'Bacia_Mecuburi', key: 'bacia_mecuburi', color: '#1ABC9C', name: 'Bacia do Mecuburi' },   // Turquesa forte
                { file: 'Megarruma', key: 'megarruma', color: '#E91E63', name: 'Bacia Megarruma' },                // Rosa forte
                { file: 'Barragens', key: 'barragens', color: '#8B4513', name: 'Barragens' },
                { file: 'Albufeiras', key: 'albufeiras', color: '#4169E1', name: 'Albufeiras' },
                { file: 'DGBL', key: 'dgbl', color: '#8B0000', name: 'Divisores Bacia Lúrio' },
                { file: 'DGBLic', key: 'dgblic', color: '#B22222', name: 'Divisores Bacia Licungo' },
                { file: 'DGBM', key: 'dgbm', color: '#A0522D', name: 'Divisores Bacia Messalo' },
                { file: 'DGBR', key: 'dgbr', color: '#800080', name: 'Divisores Bacia Rovuma' }
            ];

            let loadedCount = 0;
            const totalShapes = shapesReais.length;

            // Carregar cada shape real sequencialmente para evitar sobrecarga
            shapesReais.forEach((shape, index) => {
                setTimeout(() => {
                    carregarShapeReal(shape.file, shape.key, shape.color, shape.name)
                        .then(layer => {
                            if (layer) {
                                loadedCount++;
                                console.log(`📊 Progresso: ${loadedCount}/${totalShapes} shapes REAIS carregados`);

                                // Atualizar contador na legenda
                                const totalLayersElement = document.getElementById('total-layers');
                                if (totalLayersElement) {
                                    totalLayersElement.textContent = loadedCount;
                                }

                                if (loadedCount === totalShapes) {
                                    console.log('🎉 TODOS OS SHAPES REAIS CARREGADOS!');
                                    console.log('🗺️ Mapa GIS completo com geometrias originais!');

                                    // Ajustar zoom para mostrar toda a região
                                    setTimeout(() => {
                                        const allLayers = Object.values(allShapeLayers).filter(l => l);
                                        if (allLayers.length > 0) {
                                            const group = new L.featureGroup(allLayers);
                                            mapa.fitBounds(group.getBounds().pad(0.05));
                                        }
                                    }, 2000);

                                    // Criar controle de filtros avançado
                                    criarControlesFiltrosProfissionais();
                                }
                            }
                        });
                }, index * 500); // Delay entre shapes para melhor performance
            });
        }

        // Função para criar controles de filtros profissionais
        function criarControlesFiltrosProfissionais() {
            console.log('🔧 Criando controles de filtros profissionais...');

            // Remover controle antigo se existir
            if (typeof layerControl !== 'undefined') {
                mapa.removeControl(layerControl);
            }

            // Organizar camadas por categoria
            const categorias = {
                "🏛️ Shapes ARA Norte (Profissionais)": {},
                "🌊 Rios (Shapes)": {},
                "💧 Bacias Hidrográficas": {
                    "💧 Bacia Rovuma": allShapeLayers.bacia_rovuma,
                    "💧 Bacia Messalo": allShapeLayers.bacia_messalo,
                    "💧 Bacia Lúrio": allShapeLayers.bacia_licungo,
                    "💧 Bacia Monapo": allShapeLayers.bacia_monapo,
                    "💧 Bacia Meluli": allShapeLayers.bacia_meluli,
                    "💧 Bacia Mecuburi": allShapeLayers.bacia_mecuburi,
                    "💧 Bacia Megarruma": allShapeLayers.megarruma
                },
                "🌄 Divisores Simulados": {
                    "🌄 Divisores Bacia Lúrio": allShapeLayers.dgbl,
                    "🌄 Divisores Bacia Licungo": allShapeLayers.dgblic,
                    "🌄 Divisores Bacia Messalo": allShapeLayers.dgbm,
                    "🌄 Divisores Bacia Rovuma": allShapeLayers.dgbr
                },
                "🏗️ Infraestrutura": {
                    "🏗️ Barragens": allShapeLayers.barragens,
                    "🏗️ Albufeiras": allShapeLayers.albufeiras
                },
                "📊 Dados de Base": {
                    "🏗️ Barragens": markersBarragens,
                    "📊 Estações": markersEstacoes,
                    "🌊 Bacias": markersBacias
                }
            };

            // Adicionar camadas específicas
            if (allShapeLayers.ara_norte_ip) {
                categorias["🏛️ Shapes ARA Norte (Profissionais)"]["🏛️ Área ARA Norte"] = allShapeLayers.ara_norte_ip;
            }

            if (allShapeLayers.rios) {
                categorias["🌊 Rios (Shapes)"]["🌊 Rios da ARA Norte"] = allShapeLayers.rios;
            }

            // Filtrar camadas que existem
            Object.keys(categorias).forEach(cat => {
                Object.keys(categorias[cat]).forEach(layer => {
                    if (!categorias[cat][layer]) {
                        delete categorias[cat][layer];
                    }
                });
            });

            // Criar novo controle organizado
            layerControl = L.control.layers(baseLayers, {}, {
                collapsed: false,
                position: 'topright'
            });

            // Adicionar todas as camadas organizadas
            Object.keys(categorias).forEach(categoria => {
                Object.keys(categorias[categoria]).forEach(layerName => {
                    if (categorias[categoria][layerName]) {
                        layerControl.addOverlay(categorias[categoria][layerName], layerName);
                    }
                });
            });

            layerControl.addTo(mapa);

            // Implementar eventos dos filtros profissionais
            implementarEventosFiltrosProfissionais();

            console.log('✅ Controles de filtros profissionais criados!');
        }


        // Função para implementar eventos dos filtros simplificados
        function implementarEventosFiltrosProfissionais() {
            const filterBacia = document.getElementById('filter-bacia-hidro');
            const filterProvincia = document.getElementById('filter-provincia');
            const filterInfraestrutura = document.getElementById('filter-infraestrutura-tipo');
            const filterTipoEstacao = document.getElementById('filter-tipo-estacao');
            const filterSearch = document.getElementById('filter-search');
            const btnAplicar = document.getElementById('btn-aplicar-filtros');
            const btnLimpar = document.getElementById('btn-limpar-filtros');
            const btnMostrarTudo = document.getElementById('btn-mostrar-tudo');
            const statusDiv = document.getElementById('filter-status');

            // Mapeamento de camadas
            const layerMapping = {
                // Bacias
                'bacia_rovuma': 'bacia_rovuma',
                'bacia_messalo': 'bacia_messalo',
                'bacia_licungo': 'bacia_licungo',
                'bacia_monapo': 'bacia_monapo',
                'bacia_meluli': 'bacia_meluli',
                'bacia_mecuburi': 'bacia_mecuburi',
                'megarruma': 'megarruma',

                // Infraestrutura
                'barragens': 'barragens',
                'albufeiras': 'albufeiras',
                'rios': 'rios',
                'divisores': ['dgbl', 'dgblic', 'dgbm', 'dgbr'],

                // Área ARA Norte
                'ara_norte_ip': 'ara_norte_ip'
            };

            function aplicarFiltrosCompletos() {
                const baciaVal = filterBacia?.value || '';
                const provinciaVal = filterProvincia?.value || '';
                const infraVal = filterInfraestrutura?.value || '';
                const tipoEstacaoVal = filterTipoEstacao?.value || '';
                const searchVal = filterSearch?.value.toLowerCase() || '';

                console.log('🔍 Aplicando filtros:', {
                    bacia: baciaVal,
                    provincia: provinciaVal,
                    infra: infraVal,
                    tipoEstacao: tipoEstacaoVal,
                    search: searchVal
                });

                // Ocultar todas as camadas shape primeiro
                Object.values(allShapeLayers).forEach(layer => {
                    if (layer && mapa.hasLayer(layer)) {
                        mapa.removeLayer(layer);
                    }
                });

                // Limpar marcadores
                markersBarragens.clearLayers();
                markersEstacoes.clearLayers();
                markersBacias.clearLayers();

                let contadores = { barragens: 0, estacoes: 0, bacias: 0, shapes: 0 };

                // Filtrar marcadores de barragens
                dadosBarragens.forEach(function(barragem) {
                    let mostrar = true;

                    // Filtro de província (se barragem tem essa propriedade)
                    if (provinciaVal && barragem.provincia && barragem.provincia !== provinciaVal) mostrar = false;

                    // Filtro de pesquisa
                    if (searchVal && !barragem.nome.toLowerCase().includes(searchVal) && !barragem.codigo.toLowerCase().includes(searchVal)) mostrar = false;

                        if (mostrar && barragem.latitude && barragem.longitude) {
                            var marker = L.marker([barragem.latitude, barragem.longitude], {
                                icon: criarIcone('barragem', '#2563eb')
                            }).bindPopup(`
                                <div class="p-3">
                                    <h4 class="font-bold text-blue-800 mb-2">${barragem.nome}</h4>
                                    <p><strong>Código:</strong> ${barragem.codigo}</p>
                                    <p><strong>Tipo:</strong> Barragem</p>
                                    ${barragem.altura ? `<p><strong>Altura:</strong> ${barragem.altura}m</p>` : ''}
                                    ${barragem.cota_atual ? `<p><strong>Cota Atual:</strong> ${barragem.cota_atual}m</p>` : ''}
                                    ${barragem.percentagem_enchimento ? `<p><strong>Enchimento:</strong> ${barragem.percentagem_enchimento}%</p>` : ''}
                                    <div class="mt-3 pt-2 border-t border-gray-200">
                                        <a href="/barragens/${barragem.id}" class="inline-flex items-center px-3 py-2 bg-blue-600 hover:bg-blue-700 text-white text-sm font-medium rounded-lg transition-colors duration-200">
                                            <i class="fas fa-arrow-right mr-2"></i>Ver Detalhes
                                        </a>
                                    </div>
                                </div>
                            `);
                            markersBarragens.addLayer(marker);
                            contadores.barragens++;
                        }
                    });
                    markersBarragens.addTo(mapa);

                // Filtrar marcadores de estações
                dadosEstacoes.forEach(function(estacao) {
                    let mostrar = true;

                    // Filtro de província
                    if (provinciaVal && estacao.provincia !== provinciaVal) mostrar = false;

                    // Filtro de tipo de estação
                    if (tipoEstacaoVal && estacao.tipo !== tipoEstacaoVal) mostrar = false;

                    // Filtro de pesquisa
                    if (searchVal && !estacao.nome.toLowerCase().includes(searchVal) && !estacao.codigo.toLowerCase().includes(searchVal)) mostrar = false;

                        if (mostrar && estacao.latitude && estacao.longitude) {
                            var cor = estacao.tipo === 'hidrometrica' ? '#16a34a' :
                                     estacao.tipo === 'pluviometrica' ? '#0891b2' : '#7c3aed';

                            var marker = L.marker([estacao.latitude, estacao.longitude], {
                                icon: criarIcone('estacao', cor)
                            }).bindPopup(`
                                <div class="p-3">
                                    <h4 class="font-bold text-green-800 mb-2">${estacao.nome}</h4>
                                    <p><strong>Código:</strong> ${estacao.codigo}</p>
                                    <p><strong>Tipo:</strong> ${estacao.tipo}</p>
                                    <p><strong>Estado:</strong> ${estacao.estado}</p>
                                    ${estacao.provincia ? `<p><strong>Província:</strong> ${estacao.provincia}</p>` : ''}
                                    <div class="mt-3 pt-2 border-t border-gray-200">
                                        <a href="/estacoes/${estacao.id}" class="inline-flex items-center px-3 py-2 bg-green-600 hover:bg-green-700 text-white text-sm font-medium rounded-lg transition-colors duration-200">
                                            <i class="fas fa-arrow-right mr-2"></i>Ver Detalhes
                                        </a>
                                    </div>
                                </div>
                            `);
                            markersEstacoes.addLayer(marker);
                            contadores.estacoes++;
                        }
                    });
                    markersEstacoes.addTo(mapa);

                // Filtrar marcadores de bacias
                dadosBacias.forEach(function(bacia) {
                    let mostrar = true;

                    // Filtro de bacia (se aplicável aos marcadores de bacia)
                    if (baciaVal && bacia.codigo && !bacia.codigo.toLowerCase().includes(baciaVal)) mostrar = false;

                    // Filtro de pesquisa
                    if (searchVal && !bacia.nome.toLowerCase().includes(searchVal) && !bacia.codigo.toLowerCase().includes(searchVal)) mostrar = false;

                        if (mostrar && bacia.latitude && bacia.longitude) {
                            var marker = L.marker([bacia.latitude, bacia.longitude], {
                                icon: criarIcone('bacia', '#0891b2')
                            }).bindPopup(`
                                <div class="p-3">
                                    <h4 class="font-bold text-cyan-800 mb-2">${bacia.nome}</h4>
                                    <p><strong>Código:</strong> ${bacia.codigo}</p>
                                    <p><strong>Tipo:</strong> Bacia Hidrográfica</p>
                                    ${bacia.area_km2 ? `<p><strong>Área:</strong> ${bacia.area_km2} km²</p>` : ''}
                                    ${bacia.provincias_abrangidas ? `<p><strong>Províncias:</strong> ${bacia.provincias_abrangidas}</p>` : ''}
                                    <div class="mt-3 pt-2 border-t border-gray-200">
                                        <a href="/bacias/${bacia.id}" class="inline-flex items-center px-3 py-2 bg-cyan-600 hover:bg-cyan-700 text-white text-sm font-medium rounded-lg transition-colors duration-200">
                                            <i class="fas fa-arrow-right mr-2"></i>Ver Detalhes
                                        </a>
                                    </div>
                                </div>
                            `);
                            markersBacias.addLayer(marker);
                            contadores.bacias++;
                        }
                    });
                    markersBacias.addTo(mapa);

                // Aplicar filtros de shapes GIS - Carregar sob demanda
                // Mapeamento de keys para nomes de arquivos
                const shapeFiles = {
                    'bacia_rovuma': { file: 'Bacia_Rovuma', color: '#E74C3C', name: 'Bacia do Rovuma' },
                    'bacia_messalo': { file: 'Bacia_Messalo', color: '#2ECC71', name: 'Bacia do Messalo' },
                    'bacia_licungo': { file: 'Bacia_Licungo', color: '#3498DB', name: 'Bacia do Lúrio' },
                    'bacia_monapo': { file: 'Bacia_Monapo', color: '#F39C12', name: 'Bacia do Monapo' },
                    'bacia_meluli': { file: 'Bacia_Meluli', color: '#9B59B6', name: 'Bacia do Meluli' },
                    'bacia_mecuburi': { file: 'Bacia_Mecuburi', color: '#1ABC9C', name: 'Bacia do Mecuburi' },
                    'megarruma': { file: 'Megarruma', color: '#E91E63', name: 'Bacia Megarruma' },
                    'barragens': { file: 'Barragens', color: '#8B4513', name: 'Barragens' },
                    'albufeiras': { file: 'Albufeiras', color: '#4169E1', name: 'Albufeiras' },
                    'rios': { file: 'Rios', color: '#0066FF', name: 'Rios da ARA Norte' },
                    'dgbl': { file: 'DGBL', color: '#8B0000', name: 'Divisores Bacia Lúrio' },
                    'dgblic': { file: 'DGBLic', color: '#B22222', name: 'Divisores Bacia Licungo' },
                    'dgbm': { file: 'DGBM', color: '#A0522D', name: 'Divisores Bacia Messalo' },
                    'dgbr': { file: 'DGBR', color: '#800080', name: 'Divisores Bacia Rovuma' }
                };

                // Aplicar filtro de bacia
                if (baciaVal) {
                    const layerKey = layerMapping[baciaVal];
                    if (layerKey) {
                        // Se não existe, carregar sob demanda
                        if (!allShapeLayers[layerKey] && shapeFiles[layerKey]) {
                            const config = shapeFiles[layerKey];
                            console.log(`🔄 Carregando bacia: ${config.name}...`);
                            carregarShapeReal(config.file, layerKey, config.color, config.name)
                                .then(layer => {
                                    if (layer) {
                                        layer.addTo(mapa);
                                        contadores.shapes++;
                                        if (statusDiv) {
                                            statusDiv.innerHTML = `<i class="fas fa-check-circle mr-1 text-green-500"></i>✅ ${config.name} carregada!`;
                                        }
                                    }
                                });
                        } else if (allShapeLayers[layerKey]) {
                            allShapeLayers[layerKey].addTo(mapa);
                            contadores.shapes++;
                        }
                    }
                }

                // Aplicar filtro de infraestrutura
                if (infraVal) {
                    const layerKeys = layerMapping[infraVal];
                    const keysArray = Array.isArray(layerKeys) ? layerKeys : [layerKeys];

                    keysArray.forEach(key => {
                        if (!allShapeLayers[key] && shapeFiles[key]) {
                            // Carregar sob demanda
                            const config = shapeFiles[key];
                            console.log(`🔄 Carregando: ${config.name}...`);
                            carregarShapeReal(config.file, key, config.color, config.name)
                                .then(layer => {
                                    if (layer) {
                                        layer.addTo(mapa);
                                        contadores.shapes++;
                                        if (statusDiv) {
                                            statusDiv.innerHTML = `<i class="fas fa-check-circle mr-1 text-green-500"></i>✅ ${config.name} carregado!`;
                                        }
                                    }
                                });
                        } else if (allShapeLayers[key]) {
                            allShapeLayers[key].addTo(mapa);
                            contadores.shapes++;
                        }
                    });
                }

                // Atualizar status
                if (statusDiv) {
                    const statusParts = [];
                    if (contadores.barragens > 0) statusParts.push(`🏗️ ${contadores.barragens} barragens`);
                    if (contadores.estacoes > 0) statusParts.push(`📊 ${contadores.estacoes} estações`);
                    if (contadores.bacias > 0) statusParts.push(`🌊 ${contadores.bacias} bacias`);
                    if (contadores.shapes > 0) statusParts.push(`🗺️ ${contadores.shapes} shapes`);

                    const statusText = statusParts.length > 0
                        ? `📊 Mostrando: ${statusParts.join(', ')}`
                        : '📊 Nenhum resultado encontrado';
                    statusDiv.innerHTML = `<i class="fas fa-info-circle mr-1 text-blue-500"></i>${statusText}`;
                }
            }

            // Eventos dos botões
            if (btnAplicar) {
                btnAplicar.addEventListener('click', aplicarFiltrosCompletos);
            }

            if (btnLimpar) {
                btnLimpar.addEventListener('click', () => {
                    // Limpar valores dos filtros
                    if (filterBacia) filterBacia.value = '';
                    if (filterProvincia) filterProvincia.value = '';
                    if (filterInfraestrutura) filterInfraestrutura.value = '';
                    if (filterTipoEstacao) filterTipoEstacao.value = '';
                    if (filterSearch) filterSearch.value = '';

                    // Ocultar todas as camadas shape
                    Object.values(allShapeLayers).forEach(layer => {
                        if (layer && mapa.hasLayer(layer)) {
                            mapa.removeLayer(layer);
                        }
                    });

                    // Limpar marcadores
                    markersBarragens.clearLayers();
                    markersEstacoes.clearLayers();
                    markersBacias.clearLayers();

                    if (statusDiv) {
                        statusDiv.innerHTML = '<i class="fas fa-info-circle mr-1 text-blue-500"></i>📊 Todos os filtros limpos';
                    }
                });
            }

            if (btnMostrarTudo) {
                btnMostrarTudo.addEventListener('click', () => {
                    // Mostrar todos os shapes
                    Object.values(allShapeLayers).forEach(layer => {
                        if (layer && !mapa.hasLayer(layer)) {
                            layer.addTo(mapa);
                        }
                    });

                    // Re-adicionar todos os marcadores
                    aplicarFiltrosCompletos();

                    if (statusDiv) {
                        statusDiv.innerHTML = '<i class="fas fa-info-circle mr-1 text-blue-500"></i>📊 Todas as camadas visíveis';
                    }
                });
            }

            // Auto-aplicar ao mudar qualquer filtro
            [filterBacia, filterProvincia, filterInfraestrutura, filterTipoEstacao].forEach(filter => {
                if (filter) {
                    filter.addEventListener('change', aplicarFiltrosCompletos);
                }
            });

            // Aplicar pesquisa com delay
            if (filterSearch) {
                let searchTimeout;
                filterSearch.addEventListener('input', () => {
                    clearTimeout(searchTimeout);
                    searchTimeout = setTimeout(aplicarFiltrosCompletos, 500);
                });
            }
        }

        // Função fallback (dados simulados)
        function carregarShapeARAFallback(shapeName, layerKey, cor) {
            return fetch(`/mapas/shapes/${shapeName}`)
                .then(response => response.json())
                .then(geojson => {
                    if (shapeLayersARA[layerKey]) {
                        mapa.removeLayer(shapeLayersARA[layerKey]);
                    }

                    shapeLayersARA[layerKey] = L.geoJSON(geojson, {
                        style: function(feature) {
                            return {
                                color: cor,
                                weight: 2,
                                opacity: 0.8,
                                fillOpacity: 0.2
                            };
                        },
                        onEachFeature: function(feature, layer) {
                            let popupContent = `<div class="p-2">
                                <h4 class="font-bold" style="color: ${cor};">${feature.properties.name || shapeName}</h4>
                                <p class="text-xs mt-1"><em>📊 Dados simulados (fallback)</em></p></div>`;

                            layer.bindPopup(popupContent);
                        }
                    }).addTo(mapa);

                    console.log(`✅ Shape ${shapeName} carregado (fallback)`);
                    return shapeLayersARA[layerKey];
                })
                .catch(error => {
                    console.error(`❌ Erro no fallback ${shapeName}:`, error);
                });
        }

        // Função para adicionar listener com verificação
        function adicionarListenerSeguro(elementId, callback) {
            const element = document.getElementById(elementId);
            if (element) {
                element.addEventListener('change', callback);
            } else {
                console.warn(`⚠️ Elemento ${elementId} não encontrado`);
            }
        }

        // Controles para shapes profissionais da ARA Norte
        adicionarListenerSeguro('layer-ara-norte', function() {
            if (this.checked) {
                carregarShapeARAReal('ara_norte', 'araNorte', '#ff7800');
            } else if (shapeLayersARA.araNorte) {
                mapa.removeLayer(shapeLayersARA.araNorte);
                shapeLayersARA.araNorte = null;
            }
        });

        adicionarListenerSeguro('layer-rios-shapes', function() {
            if (this.checked) {
                carregarShapeARAReal('rios', 'riosShapes', '#0066ff');
            } else if (shapeLayersARA.riosShapes) {
                mapa.removeLayer(shapeLayersARA.riosShapes);
                shapeLayersARA.riosShapes = null;
            }
        });

        adicionarListenerSeguro('layer-bacia-rovuma', function() {
            if (this.checked) {
                carregarShapeARAReal('bacia_rovuma', 'baciaRovuma', '#0066cc');
            } else if (shapeLayersARA.baciaRovuma) {
                mapa.removeLayer(shapeLayersARA.baciaRovuma);
                shapeLayersARA.baciaRovuma = null;
            }
        });

        adicionarListenerSeguro('layer-bacia-messalo', function() {
            if (this.checked) {
                carregarShapeARAReal('bacia_messalo', 'baciaMessalo', '#4169e1');
            } else if (shapeLayersARA.baciaMessalo) {
                mapa.removeLayer(shapeLayersARA.baciaMessalo);
                shapeLayersARA.baciaMessalo = null;
            }
        });

        adicionarListenerSeguro('layer-bacia-lurio', function() {
            if (this.checked) {
                carregarShapeARAReal('dgbl', 'baciaLurio', '#00ced1');
            } else if (shapeLayersARA.baciaLurio) {
                mapa.removeLayer(shapeLayersARA.baciaLurio);
                shapeLayersARA.baciaLurio = null;
            }
        });

        // Debug: verificar se libraries estão carregadas
        console.log('🔍 Verificando libraries...');
        console.log('L (Leaflet) disponível:', typeof L !== 'undefined');
        console.log('fetch disponível:', typeof fetch !== 'undefined');

        // 🗺️ NÃO CARREGAR SHAPES AUTOMATICAMENTE - Carregar sob demanda via filtros
        console.log('✅ Dashboard pronto! Use os filtros para carregar as bacias.');
        console.log('📊 Shapes serão carregados apenas quando solicitados nos filtros.');
        // carregarTodosOsShapesReais(); // DESATIVADO - muito lento

        // Inicializar filtros imediatamente
        implementarEventosFiltrosProfissionais();

        adicionarListenerSeguro('layer-watersheds', function() {
            if (this.checked) {
                console.log('Ativando divisores de bacia...');

                if (!shapeLayersARA.watersheds) {
                    // Simular camada de bacias com polígonos
                    shapeLayersARA.watersheds = L.layerGroup();

                    // 12 Bacias Hidrográficas Oficiais da ARA Norte (de sul para norte)
                    var baciasARANorte = [
                        {
                            nome: 'Licungo',
                            coords: [[-16.2, 36.8], [-16.5, 37.4], [-16.8, 37.8], [-16.5, 38.2], [-16.0, 37.8], [-15.8, 37.2], [-16.2, 36.8]],
                            cor: '#8b4513',
                            info: 'Limite sul da ARA Norte - Nampula/Zambézia'
                        },
                        {
                            nome: 'Melela',
                            coords: [[-16.0, 37.8], [-16.2, 38.2], [-16.0, 38.6], [-15.8, 38.4], [-15.6, 38.0], [-16.0, 37.8]],
                            cor: '#9932cc',
                            info: 'Pequena bacia costeira - Nampula'
                        },
                        {
                            nome: 'Molocuè',
                            coords: [[-15.8, 38.0], [-16.0, 38.4], [-15.8, 38.8], [-15.6, 38.6], [-15.4, 38.2], [-15.8, 38.0]],
                            cor: '#ff4500',
                            info: 'Bacia central de Nampula'
                        },
                        {
                            nome: 'Ligonha',
                            coords: [[-15.4, 37.6], [-15.8, 38.2], [-15.6, 38.8], [-15.2, 39.2], [-14.8, 38.6], [-15.0, 38.0], [-15.4, 37.6]],
                            cor: '#32cd32',
                            info: 'Atravessa Nampula de oeste para leste'
                        },
                        {
                            nome: 'Meluli',
                            coords: [[-15.0, 38.6], [-15.2, 39.0], [-14.8, 39.4], [-14.6, 39.0], [-14.8, 38.6], [-15.0, 38.6]],
                            cor: '#1e90ff',
                            info: 'Pequena bacia entre Ligonha e Monapo'
                        },
                        {
                            nome: 'Monapo',
                            coords: [[-14.6, 38.8], [-14.8, 39.2], [-14.4, 39.6], [-14.2, 39.2], [-14.4, 38.8], [-14.6, 38.8]],
                            cor: '#ff69b4',
                            info: 'Bacia do distrito de Monapo - Nampula'
                        },
                        {
                            nome: 'Mecuburi',
                            coords: [[-14.2, 38.6], [-14.4, 39.0], [-14.0, 39.4], [-13.8, 39.0], [-14.0, 38.6], [-14.2, 38.6]],
                            cor: '#ffd700',
                            info: 'Distrito de Mecuburi - Nampula'
                        },
                        {
                            nome: 'Lúrio',
                            coords: [[-13.6, 37.8], [-14.2, 38.4], [-14.6, 39.0], [-15.0, 39.6], [-14.8, 40.0], [-14.4, 40.2], [-13.8, 39.8], [-13.4, 39.2], [-13.2, 38.6], [-13.6, 37.8]],
                            cor: '#00ced1',
                            info: 'Maior bacia de Nampula - Partilhada com Cabo Delgado'
                        },
                        {
                            nome: 'Megaruma',
                            coords: [[-13.4, 39.4], [-13.8, 39.8], [-13.6, 40.2], [-13.2, 40.0], [-13.0, 39.6], [-13.4, 39.4]],
                            cor: '#dc143c',
                            info: 'Menor bacia (5490 km²) - Sul de Cabo Delgado'
                        },
                        {
                            nome: 'Montepuez',
                            coords: [[-12.8, 38.8], [-13.2, 39.4], [-12.8, 40.0], [-12.4, 39.6], [-12.6, 39.0], [-12.8, 38.8]],
                            cor: '#ff6347',
                            info: 'Centro de Cabo Delgado - Distrito de Montepuez'
                        },
                        {
                            nome: 'Messalo',
                            coords: [[-12.0, 37.8], [-12.8, 38.6], [-12.6, 39.8], [-12.2, 40.4], [-11.8, 40.0], [-11.6, 39.2], [-11.8, 38.4], [-12.0, 37.8]],
                            cor: '#4169e1',
                            info: 'Grande parte de Cabo Delgado e Niassa'
                        },
                        {
                            nome: 'Rovuma',
                            coords: [[-10.47, 40.44], [-10.8, 40.0], [-11.2, 39.6], [-11.8, 39.2], [-12.4, 38.8], [-12.8, 38.4], [-13.2, 38.0], [-13.0, 37.6], [-12.6, 37.2], [-12.0, 37.0], [-11.4, 37.4], [-10.8, 38.0], [-10.6, 38.8], [-10.5, 39.6], [-10.47, 40.44]],
                            cor: '#0066cc',
                            info: 'Maior bacia (99.524 km²) - Fronteira com Tanzânia'
                        }
                    ];

                    baciasARANorte.forEach(function(bacia) {
                        var polygon = L.polygon(bacia.coords, {
                            color: bacia.cor,
                            weight: 2,
                            fillOpacity: 0.15,
                            opacity: 0.8
                        }).bindPopup(`
                            <div class="p-2">
                                <h4 class="font-bold" style="color: ${bacia.cor};">Bacia do ${bacia.nome}</h4>
                                <p class="text-xs">${bacia.info}</p>
                                <p class="text-xs mt-1"><strong>ARA Norte</strong> - Gestão oficial</p>
                            </div>
                        `);
                        shapeLayersARA.watersheds.addLayer(polygon);
                    });


                    shapeLayersARA.watersheds.addTo(mapa);
                    console.log('Divisores de bacia carregados');
                }
            } else {
                if (shapeLayersARA.watersheds) {
                    mapa.removeLayer(shapeLayersARA.watersheds);
                    shapeLayersARA.watersheds = null;
                    console.log('Divisores de bacia removidos');
                }
            }
        });

        // Nota: layer-rivers foi substituído por layer-rios-shapes (carregamento de shapes reais)

        // Removidos botões não funcionais: Reset, Exportar, Medir, Análise
        // Interface focada apenas em funcionalidades operacionais


        // Panel de informações
        adicionarListenerSeguro('close-info', function() {
            const infoPanelFinal = document.getElementById('info-panel');
            if (infoPanelFinal) {
                infoPanelFinal.classList.add('hidden');
            }
        });

        console.log('Mapa GIS carregado com sucesso!');
        console.log(`Marcadores carregados: ${counts.barragens} barragens, ${counts.estacoes} estações, ${counts.bacias} bacias`);

    } catch (error) {
        console.error('Erro ao carregar mapa GIS:', error);
        document.getElementById('mapa-principal').innerHTML =
            '<div class="flex items-center justify-center h-full text-red-600">Erro ao carregar mapa GIS: ' + error.message + '</div>';
    }
});
</script>
@endsection