@extends('layouts.app')

@section('title', 'Gestão de Barragens')

@section('content')
<div class="space-y-6">
    <!-- Header -->
    <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between">
            <div>
                <h1 class="text-2xl font-bold text-gray-800">
                    <i class="fas fa-dam mr-3 text-blue-600"></i>
                    Gestão de Barragens
                </h1>
                <p class="text-gray-600 mt-2">Gerir todas as barragens do sistema hidrológico</p>
            </div>
            <div class="flex space-x-4">
                <a href="{{ route('barragens.create') }}" 
                   class="bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700 transition-colors">
                    <i class="fas fa-plus mr-2"></i>
                    Nova Barragem
                </a>
                <a href="{{ route('importacao.index') }}" 
                   class="bg-green-600 text-white px-4 py-2 rounded-md hover:bg-green-700 transition-colors">
                    <i class="fas fa-upload mr-2"></i>
                    Importar Dados
                </a>
            </div>
        </div>
    </div>

    <!-- Filters -->
    <div class="bg-white rounded-lg shadow p-6">
        <form method="GET" action="{{ route('barragens.index') }}" class="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div>
                <label for="search" class="block text-sm font-medium text-gray-700 mb-1">Pesquisar</label>
                <input type="text" id="search" name="search" value="{{ request('search') }}" 
                       placeholder="Nome ou código..."
                       class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
            </div>
            
            <div>
                <label for="provincia" class="block text-sm font-medium text-gray-700 mb-1">Província</label>
                <select id="provincia" name="provincia" 
                        class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                    <option value="">Todas as Províncias</option>
                    <option value="Nampula" {{ request('provincia') == 'Nampula' ? 'selected' : '' }}>Nampula</option>
                    <option value="Zambézia" {{ request('provincia') == 'Zambézia' ? 'selected' : '' }}>Zambézia</option>
                    <option value="Niassa" {{ request('provincia') == 'Niassa' ? 'selected' : '' }}>Niassa</option>
                    <option value="Cabo Delgado" {{ request('provincia') == 'Cabo Delgado' ? 'selected' : '' }}>Cabo Delgado</option>
                </select>
            </div>
            
            <div>
                <label for="estado" class="block text-sm font-medium text-gray-700 mb-1">Estado</label>
                <select id="estado" name="estado" 
                        class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                    <option value="">Todos os Estados</option>
                    <option value="activa" {{ request('estado') == 'activa' ? 'selected' : '' }}>Ativa</option>
                    <option value="inactiva" {{ request('estado') == 'inactiva' ? 'selected' : '' }}>Inativa</option>
                    <option value="manutencao" {{ request('estado') == 'manutencao' ? 'selected' : '' }}>Manutenção</option>
                </select>
            </div>
            
            <div class="flex items-end space-x-2">
                <button type="submit" 
                        class="bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700 transition-colors">
                    <i class="fas fa-search mr-2"></i>
                    Filtrar
                </button>
                <a href="{{ route('barragens.index') }}" 
                   class="bg-gray-500 text-white px-4 py-2 rounded-md hover:bg-gray-600 transition-colors">
                    <i class="fas fa-times mr-2"></i>
                    Limpar
                </a>
            </div>
        </form>
    </div>

    <!-- Statistics Cards -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-blue-100 text-blue-600">
                    <i class="fas fa-dam text-xl"></i>
                </div>
                <div class="ml-4">
                    <h3 class="text-sm font-medium text-gray-500">Total de Barragens</h3>
                    <p class="text-2xl font-bold text-gray-900">{{ $barragens->total() }}</p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-green-100 text-green-600">
                    <i class="fas fa-check-circle text-xl"></i>
                </div>
                <div class="ml-4">
                    <h3 class="text-sm font-medium text-gray-500">Ativas</h3>
                    <p class="text-2xl font-bold text-green-600">
                        {{ $barragens->where('estado', 'activa')->count() }}
                    </p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-yellow-100 text-yellow-600">
                    <i class="fas fa-tools text-xl"></i>
                </div>
                <div class="ml-4">
                    <h3 class="text-sm font-medium text-gray-500">Manutenção</h3>
                    <p class="text-2xl font-bold text-yellow-600">
                        {{ $barragens->where('estado', 'manutencao')->count() }}
                    </p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-red-100 text-red-600">
                    <i class="fas fa-exclamation-triangle text-xl"></i>
                </div>
                <div class="ml-4">
                    <h3 class="text-sm font-medium text-gray-500">Com Alertas</h3>
                    <p class="text-2xl font-bold text-red-600">
                        {{ $barragens->filter(function($b) { return $b->alertas->where('estado', 'activo')->count() > 0; })->count() }}
                    </p>
                </div>
            </div>
        </div>
    </div>

    <!-- Barragens List -->
    <div class="bg-white rounded-lg shadow">
        <div class="p-6 border-b border-gray-200">
            <h3 class="text-lg font-semibold text-gray-800">
                Lista de Barragens
                <span class="text-sm text-gray-500 font-normal">
                    ({{ $barragens->firstItem() ?? 0 }}-{{ $barragens->lastItem() ?? 0 }} de {{ $barragens->total() }})
                </span>
            </h3>
        </div>
        
        @if($barragens->count() > 0)
            <div class="overflow-x-auto">
                <table class="w-full">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Barragem
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Localização
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Última Leitura
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Estado
                            </th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Status
                            </th>
                            <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Ações
                            </th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        @foreach($barragens as $barragem)
                        <tr class="hover:bg-gray-50">
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="flex items-center">
                                    <div class="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center">
                                        <i class="fas fa-dam text-blue-600"></i>
                                    </div>
                                    <div class="ml-4">
                                        <div class="text-sm font-medium text-gray-900">
                                            <a href="{{ route('barragens.show', $barragem) }}" 
                                               class="hover:text-blue-600">
                                                {{ $barragem->nome }}
                                            </a>
                                        </div>
                                        <div class="text-sm text-gray-500">{{ $barragem->codigo }}</div>
                                        @if($barragem->bacia_hidrografica)
                                            <div class="text-xs text-blue-600">{{ $barragem->bacia_hidrografica->nome }}</div>
                                        @endif
                                    </div>
                                </div>
                            </td>
                            
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="text-sm text-gray-900">
                                    {{ $barragem->provincia }}
                                    @if($barragem->distrito)
                                        , {{ $barragem->distrito }}
                                    @endif
                                </div>
                                @if($barragem->latitude && $barragem->longitude)
                                    <div class="text-xs text-gray-500">
                                        {{ number_format($barragem->latitude, 4) }}°, {{ number_format($barragem->longitude, 4) }}°
                                    </div>
                                @endif
                            </td>
                            
                            <td class="px-6 py-4 whitespace-nowrap">
                                @if($barragem->ultima_leitura)
                                    <div class="text-sm text-gray-900">
                                        {{ number_format($barragem->ultima_leitura->cota_actual, 2) }}m
                                    </div>
                                    <div class="text-xs text-gray-500">
                                        {{ $barragem->ultima_leitura->data_leitura->format('d/m/Y') }}
                                    </div>
                                    @if($barragem->ultima_leitura->percentagem_enchimento)
                                        <div class="flex items-center mt-1">
                                            <div class="w-12 bg-gray-200 rounded-full h-1.5 mr-2">
                                                <div class="bg-blue-600 h-1.5 rounded-full" 
                                                     style="width: {{ min(100, $barragem->ultima_leitura->percentagem_enchimento) }}%"></div>
                                            </div>
                                            <span class="text-xs text-gray-600">
                                                {{ number_format($barragem->ultima_leitura->percentagem_enchimento, 1) }}%
                                            </span>
                                        </div>
                                    @endif
                                @else
                                    <span class="text-sm text-gray-400">Sem leituras</span>
                                @endif
                            </td>
                            
                            <td class="px-6 py-4 whitespace-nowrap">
                                @switch($barragem->estado)
                                    @case('activa')
                                        <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800">
                                            Ativa
                                        </span>
                                        @break
                                    @case('inactiva')
                                        <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-red-100 text-red-800">
                                            Inativa
                                        </span>
                                        @break
                                    @case('manutencao')
                                        <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-yellow-100 text-yellow-800">
                                            Manutenção
                                        </span>
                                        @break
                                @endswitch
                            </td>
                            
                            <td class="px-6 py-4 whitespace-nowrap">
                                @php
                                    $alertas_ativos = $barragem->alertas->where('estado', 'activo');
                                    $nivel_critico = $barragem->ultima_leitura ? $barragem->getStatusNivel($barragem->ultima_leitura->cota_actual) : 'normal';
                                @endphp
                                
                                @if($alertas_ativos->count() > 0)
                                    @php $alerta_prioridade = $alertas_ativos->sortByDesc('nivel')->first(); @endphp
                                    <div class="flex items-center">
                                        @switch($alerta_prioridade->nivel)
                                            @case('emergencia')
                                                <div class="w-2 h-2 bg-red-500 rounded-full mr-2 animate-pulse"></div>
                                                <span class="text-xs text-red-700 font-medium">Emergência</span>
                                                @break
                                            @case('alerta')
                                                <div class="w-2 h-2 bg-orange-500 rounded-full mr-2 animate-pulse"></div>
                                                <span class="text-xs text-orange-700 font-medium">Alerta</span>
                                                @break
                                            @case('atencao')
                                                <div class="w-2 h-2 bg-yellow-500 rounded-full mr-2"></div>
                                                <span class="text-xs text-yellow-700 font-medium">Atenção</span>
                                                @break
                                            @default
                                                <div class="w-2 h-2 bg-blue-500 rounded-full mr-2"></div>
                                                <span class="text-xs text-blue-700 font-medium">Informação</span>
                                        @endswitch
                                        <span class="ml-1 text-xs text-gray-500">({{ $alertas_ativos->count() }})</span>
                                    </div>
                                @else
                                    @switch($nivel_critico)
                                        @case('emergencia')
                                            <div class="flex items-center">
                                                <div class="w-2 h-2 bg-red-500 rounded-full mr-2"></div>
                                                <span class="text-xs text-red-700">Crítico</span>
                                            </div>
                                            @break
                                        @case('alerta')
                                            <div class="flex items-center">
                                                <div class="w-2 h-2 bg-yellow-500 rounded-full mr-2"></div>
                                                <span class="text-xs text-yellow-700">Atenção</span>
                                            </div>
                                            @break
                                        @default
                                            <div class="flex items-center">
                                                <div class="w-2 h-2 bg-green-500 rounded-full mr-2"></div>
                                                <span class="text-xs text-green-700">Normal</span>
                                            </div>
                                    @endswitch
                                @endif
                            </td>
                            
                            <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                <div class="flex items-center justify-end space-x-2">
                                    <a href="{{ route('barragens.show', $barragem) }}" 
                                       class="text-blue-600 hover:text-blue-900" title="Ver Detalhes">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                    <a href="{{ route('barragens.leituras', $barragem) }}" 
                                       class="text-green-600 hover:text-green-900" title="Leituras">
                                        <i class="fas fa-chart-line"></i>
                                    </a>
                                    @can('editar-barragens')
                                        <a href="{{ route('barragens.edit', $barragem) }}" 
                                           class="text-yellow-600 hover:text-yellow-900" title="Editar">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                    @endcan
                                    @can('eliminar-barragens')
                                        <button onclick="confirmDelete('{{ $barragem->id }}', '{{ $barragem->nome }}')" 
                                                class="text-red-600 hover:text-red-900" title="Eliminar">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    @endcan
                                </div>
                            </td>
                        </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <div class="px-6 py-3 border-t border-gray-200">
                {{ $barragens->links() }}
            </div>
        @else
            <div class="p-12 text-center">
                <div class="mx-auto w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mb-4">
                    <i class="fas fa-dam text-gray-400 text-3xl"></i>
                </div>
                <h3 class="text-lg font-medium text-gray-900 mb-2">Nenhuma barragem encontrada</h3>
                <p class="text-gray-500 mb-4">
                    @if(request()->anyFilled(['search', 'provincia', 'estado']))
                        Não foram encontradas barragens com os critérios de pesquisa especificados.
                    @else
                        Comece adicionando a primeira barragem ao sistema.
                    @endif
                </p>
                <div class="flex justify-center space-x-4">
                    @if(request()->anyFilled(['search', 'provincia', 'estado']))
                        <a href="{{ route('barragens.index') }}" 
                           class="bg-gray-500 text-white px-4 py-2 rounded-md hover:bg-gray-600 transition-colors">
                            <i class="fas fa-times mr-2"></i>
                            Limpar Filtros
                        </a>
                    @endif
                    <a href="{{ route('barragens.create') }}" 
                       class="bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700 transition-colors">
                        <i class="fas fa-plus mr-2"></i>
                        Adicionar Barragem
                    </a>
                </div>
            </div>
        @endif
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div id="deleteModal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 hidden">
    <div class="bg-white rounded-lg p-6 max-w-md w-full mx-4">
        <div class="flex items-center mb-4">
            <div class="w-12 h-12 bg-red-100 rounded-full flex items-center justify-center mr-4">
                <i class="fas fa-exclamation-triangle text-red-600 text-xl"></i>
            </div>
            <div>
                <h3 class="text-lg font-medium text-gray-900">Confirmar Eliminação</h3>
                <p class="text-sm text-gray-500">Esta ação não pode ser desfeita.</p>
            </div>
        </div>
        
        <p class="text-gray-700 mb-6">
            Tem certeza de que deseja eliminar a barragem <strong id="deleteItemName"></strong>?
        </p>
        
        <div class="flex justify-end space-x-4">
            <button onclick="closeDeleteModal()" 
                    class="px-4 py-2 border border-gray-300 text-gray-700 rounded-md hover:bg-gray-50">
                Cancelar
            </button>
            <form id="deleteForm" method="POST" style="display: inline;">
                @csrf
                @method('DELETE')
                <button type="submit" 
                        class="px-4 py-2 bg-red-600 text-white rounded-md hover:bg-red-700">
                    <i class="fas fa-trash mr-2"></i>
                    Eliminar
                </button>
            </form>
        </div>
    </div>
</div>

@push('scripts')
<script>
    function confirmDelete(id, name) {
        document.getElementById('deleteItemName').textContent = name;
        document.getElementById('deleteForm').action = `/barragens/${id}`;
        document.getElementById('deleteModal').classList.remove('hidden');
    }

    function closeDeleteModal() {
        document.getElementById('deleteModal').classList.add('hidden');
    }

    // Close modal on Esc key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            closeDeleteModal();
        }
    });

    // Auto-submit form on filter change
    document.addEventListener('DOMContentLoaded', function() {
        const filters = ['provincia', 'estado'];
        filters.forEach(filterId => {
            document.getElementById(filterId)?.addEventListener('change', function() {
                this.form.submit();
            });
        });
    });
</script>
@endpush
@endsection