@extends('layouts.app')

@section('title', 'Mapa da Área de Jurisdição')

@section('content')
<div class="space-y-6">
    <!-- Header -->
    <div class="bg-white rounded-lg shadow p-6">
        <div class="flex items-center justify-between">
            <div>
                <h1 class="text-2xl font-bold text-gray-800">
                    <i class="fas fa-map-marked-alt mr-3 text-blue-600"></i>
                    Mapa da Área de Jurisdição
                </h1>
                <p class="text-gray-600 mt-2">Visualização geográfica completa dos recursos hídricos por área de jurisdição</p>
            </div>
            <a href="{{ route('jurisdicao.areas') }}" class="bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700 transition-colors">
                <i class="fas fa-cog mr-2"></i>
                Gerir Áreas
            </a>
        </div>
    </div>

    <!-- Estatísticas -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0 bg-purple-100 rounded-full p-3">
                    <i class="fas fa-map text-2xl text-purple-600"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-500">Bacias Mapeadas</p>
                    <p class="text-3xl font-bold text-gray-900">{{ $bacias->count() }}</p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0 bg-green-100 rounded-full p-3">
                    <i class="fas fa-satellite-dish text-2xl text-green-600"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-500">Estações</p>
                    <p class="text-3xl font-bold text-gray-900">{{ $estacoes->count() }}</p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0 bg-blue-100 rounded-full p-3">
                    <i class="fas fa-water text-2xl text-blue-600"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-500">Barragens</p>
                    <p class="text-3xl font-bold text-gray-900">{{ $barragens->count() }}</p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0 bg-orange-100 rounded-full p-3">
                    <i class="fas fa-layer-group text-2xl text-orange-600"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-500">Áreas de Jurisdição</p>
                    <p class="text-3xl font-bold text-gray-900">{{ $areas->count() }}</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Mapa Interativo -->
    <div class="bg-white rounded-lg shadow">
        <div class="px-6 py-4 border-b border-gray-200">
            <div class="flex items-center justify-between">
                <div>
                    <h3 class="text-lg font-semibold text-gray-900">
                        <i class="fas fa-globe mr-2 text-blue-600"></i>
                        Mapa Georreferenciado
                    </h3>
                    <p class="text-sm text-gray-600 mt-1">Clique nos marcadores para ver detalhes</p>
                </div>
                <div class="flex space-x-2">
                    <button onclick="toggleLayer('barragens')" id="btn-barragens" class="px-3 py-2 bg-blue-600 text-white rounded-md text-sm hover:bg-blue-700 transition-colors">
                        <i class="fas fa-water mr-1"></i> Barragens
                    </button>
                    <button onclick="toggleLayer('estacoes')" id="btn-estacoes" class="px-3 py-2 bg-green-600 text-white rounded-md text-sm hover:bg-green-700 transition-colors">
                        <i class="fas fa-satellite-dish mr-1"></i> Estações
                    </button>
                    <button onclick="toggleLayer('bacias')" id="btn-bacias" class="px-3 py-2 bg-purple-600 text-white rounded-md text-sm hover:bg-purple-700 transition-colors">
                        <i class="fas fa-map mr-1"></i> Bacias
                    </button>
                </div>
            </div>
        </div>
        <div id="map" style="height: 700px; width: 100%;"></div>
    </div>

    <!-- Legenda -->
    <div class="bg-white rounded-lg shadow p-6">
        <h3 class="text-lg font-semibold text-gray-900 mb-4">
            <i class="fas fa-info-circle mr-2 text-blue-600"></i>
            Legenda
        </h3>
        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div class="flex items-center space-x-3">
                <i class="fas fa-water fa-2x text-blue-600"></i>
                <div>
                    <p class="font-medium text-gray-900">Barragens</p>
                    <p class="text-sm text-gray-600">Infraestruturas hídricas</p>
                </div>
            </div>
            <div class="flex items-center space-x-3">
                <i class="fas fa-satellite-dish fa-2x text-green-600"></i>
                <div>
                    <p class="font-medium text-gray-900">Estações Hidrométricas</p>
                    <p class="text-sm text-gray-600">Pontos de monitoramento</p>
                </div>
            </div>
            <div class="flex items-center space-x-3">
                <i class="fas fa-map fa-2x text-purple-600"></i>
                <div>
                    <p class="font-medium text-gray-900">Bacias Hidrográficas</p>
                    <p class="text-sm text-gray-600">Áreas de drenagem</p>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Leaflet CSS -->
<link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
<!-- Leaflet JS -->
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>

<script>
// Inicializar mapa centrado em Moçambique
const map = L.map('map').setView([-18.665695, 35.529562], 6);

// Camada base - OpenStreetMap
L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
    attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
    maxZoom: 18,
}).addTo(map);

// Grupos de camadas
const layerGroups = {
    barragens: L.layerGroup().addTo(map),
    estacoes: L.layerGroup().addTo(map),
    bacias: L.layerGroup().addTo(map)
};

// Estados das camadas
const layerStates = {
    barragens: true,
    estacoes: true,
    bacias: true
};

// Adicionar barragens
const barragens = @json($barragens);
const iconBarragem = L.divIcon({
    html: '<i class="fas fa-water fa-2x" style="color: #2563eb;"></i>',
    className: 'custom-div-icon',
    iconSize: [30, 30],
    iconAnchor: [15, 15]
});

barragens.forEach(function(barragem) {
    const marker = L.marker([barragem.latitude, barragem.longitude], {icon: iconBarragem})
        .bindPopup(`
            <div class="p-3">
                <h4 class="font-bold text-base mb-2 text-blue-900">${barragem.nome}</h4>
                <div class="space-y-1 text-sm">
                    <p><strong>Código:</strong> ${barragem.codigo}</p>
                    <p><strong>Província:</strong> ${barragem.provincia || 'N/A'}</p>
                    <p><strong>Divisão:</strong> ${barragem.divisao_administrativa ? barragem.divisao_administrativa.nome : 'Não definida'}</p>
                    <p><strong>Bacia:</strong> ${barragem.bacia_hidrografica ? barragem.bacia_hidrografica.nome : 'N/A'}</p>
                    <a href="/barragens/${barragem.id}" class="text-blue-600 hover:text-blue-800 font-medium mt-2 inline-block">
                        Ver Detalhes →
                    </a>
                </div>
            </div>
        `);
    layerGroups.barragens.addLayer(marker);
});

// Adicionar estações com cores por tipo
const estacoes = @json($estacoes);
const getIconEstacao = (tipo) => {
    const colors = {
        'hidrometrica': '#10b981',
        'pluviometrica': '#3b82f6',
        'meteorologica': '#f59e0b',
        'climatologica': '#8b5cf6',
        'evaporimetrica': '#ec4899'
    };
    return L.divIcon({
        html: `<i class="fas fa-satellite-dish fa-2x" style="color: ${colors[tipo] || '#6b7280'};"></i>`,
        className: 'custom-div-icon',
        iconSize: [30, 30],
        iconAnchor: [15, 15]
    });
};

estacoes.forEach(function(estacao) {
    const marker = L.marker([estacao.latitude, estacao.longitude], {icon: getIconEstacao(estacao.tipo)})
        .bindPopup(`
            <div class="p-3">
                <h4 class="font-bold text-base mb-2 text-green-900">${estacao.nome}</h4>
                <div class="space-y-1 text-sm">
                    <p><strong>Código:</strong> ${estacao.codigo}</p>
                    <p><strong>Tipo:</strong> <span class="capitalize">${estacao.tipo}</span></p>
                    <p><strong>Província:</strong> ${estacao.provincia || 'N/A'}</p>
                    <p><strong>Divisão:</strong> ${estacao.divisao_administrativa ? estacao.divisao_administrativa.nome : 'Não definida'}</p>
                    <p><strong>Bacia:</strong> ${estacao.bacia_hidrografica ? estacao.bacia_hidrografica.nome : 'N/A'}</p>
                    <p><strong>Estado:</strong> <span class="capitalize">${estacao.estado}</span></p>
                    <a href="/estacoes/${estacao.id}" class="text-blue-600 hover:text-blue-800 font-medium mt-2 inline-block">
                        Ver Detalhes →
                    </a>
                </div>
            </div>
        `);
    layerGroups.estacoes.addLayer(marker);
});

// Adicionar bacias
const bacias = @json($bacias);
const iconBacia = L.divIcon({
    html: '<i class="fas fa-map fa-2x" style="color: #8b5cf6;"></i>',
    className: 'custom-div-icon',
    iconSize: [30, 30],
    iconAnchor: [15, 15]
});

bacias.forEach(function(bacia) {
    // Se a bacia tem coordenadas geográficas (GeoJSON), desenhar polígono
    if(bacia.coordenadas_geograficas && bacia.coordenadas_geograficas.type === 'Polygon') {
        const coords = bacia.coordenadas_geograficas.coordinates[0].map(c => [c[1], c[0]]);
        L.polygon(coords, {
            color: '#8b5cf6',
            fillColor: '#8b5cf6',
            fillOpacity: 0.2,
            weight: 2
        }).bindPopup(`
            <div class="p-3">
                <h4 class="font-bold text-base mb-2 text-purple-900">${bacia.nome}</h4>
                <div class="space-y-1 text-sm">
                    <p><strong>Código:</strong> ${bacia.codigo}</p>
                    <p><strong>Área:</strong> ${bacia.area_km2 ? parseFloat(bacia.area_km2).toFixed(2) + ' km²' : 'N/A'}</p>
                    <p><strong>Divisão:</strong> ${bacia.divisao_administrativa ? bacia.divisao_administrativa.nome : 'Não definida'}</p>
                    <a href="/bacias/${bacia.id}" class="text-blue-600 hover:text-blue-800 font-medium mt-2 inline-block">
                        Ver Detalhes →
                    </a>
                </div>
            </div>
        `).addTo(layerGroups.bacias);
    }
    // Senão, usar marcador simples se tiver latitude/longitude
    else if(bacia.latitude && bacia.longitude) {
        L.marker([bacia.latitude, bacia.longitude], {icon: iconBacia})
            .bindPopup(`
                <div class="p-3">
                    <h4 class="font-bold text-base mb-2 text-purple-900">${bacia.nome}</h4>
                    <div class="space-y-1 text-sm">
                        <p><strong>Código:</strong> ${bacia.codigo}</p>
                        <p><strong>Área:</strong> ${bacia.area_km2 ? parseFloat(bacia.area_km2).toFixed(2) + ' km²' : 'N/A'}</p>
                        <p><strong>Divisão:</strong> ${bacia.divisao_administrativa ? bacia.divisao_administrativa.nome : 'Não definida'}</p>
                        <a href="/bacias/${bacia.id}" class="text-blue-600 hover:text-blue-800 font-medium mt-2 inline-block">
                            Ver Detalhes →
                        </a>
                    </div>
                </div>
            `).addTo(layerGroups.bacias);
    }
});

// Função para alternar camadas
function toggleLayer(layerName) {
    const btn = document.getElementById(`btn-${layerName}`);
    layerStates[layerName] = !layerStates[layerName];

    if (layerStates[layerName]) {
        layerGroups[layerName].addTo(map);
        btn.classList.remove('opacity-50');
    } else {
        map.removeLayer(layerGroups[layerName]);
        btn.classList.add('opacity-50');
    }
}

// Ajustar mapa para mostrar todos os marcadores
const allPoints = [];
barragens.forEach(b => allPoints.push([b.latitude, b.longitude]));
estacoes.forEach(e => allPoints.push([e.latitude, e.longitude]));
bacias.forEach(b => {
    if(b.latitude && b.longitude) allPoints.push([b.latitude, b.longitude]);
});

if (allPoints.length > 0) {
    const bounds = L.latLngBounds(allPoints);
    map.fitBounds(bounds, { padding: [50, 50] });
}
</script>

<style>
.custom-div-icon {
    background: transparent;
    border: none;
}
.leaflet-popup-content-wrapper {
    border-radius: 8px;
}
</style>
@endsection
