<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\BarragemController;
use App\Http\Controllers\EstacaoController;
use App\Http\Controllers\BaciaHidrograficaController;
use App\Http\Controllers\AlertaController;
use App\Http\Controllers\BoletimController;
use App\Http\Controllers\ImportacaoController;
use App\Http\Controllers\ChartsController;
use App\Http\Controllers\ConfiguracoesController;
use App\Http\Controllers\NotificationConfigController;
use App\Http\Controllers\SystemSettingsController;
use App\Http\Controllers\MapController;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "web" middleware group. Make something great!
|
*/

Route::get('/', function () {
    if (Auth::check()) {
        return redirect()->route('dashboard');
    }
    return view('welcome');
});

Route::middleware(['auth'])->group(function () {
    Route::get('/dashboard', [DashboardController::class, 'index'])->name('dashboard');
    Route::get('/dashboard-backup', [DashboardController::class, 'backup'])->name('dashboard-backup');
    
    // Barragens
    Route::resource('barragens', BarragemController::class)->parameters([
        'barragens' => 'barragem'
    ]);
    
    // Rotas para leituras de barragens
    Route::get('/barragens/{barragem}/leituras', [BarragemController::class, 'leituras'])->name('barragens.leituras');
    Route::post('/barragens/{barragem}/leituras', [BarragemController::class, 'storeLeitura'])->name('barragens.storeLeitura');
    Route::get('/barragens/{barragem}/leituras/{leitura}/edit', [BarragemController::class, 'editLeitura'])->name('barragens.leituras.edit');
    Route::put('/barragens/{barragem}/leituras/{leitura}', [BarragemController::class, 'updateLeitura'])->name('barragens.leituras.update');
    Route::delete('/barragens/{barragem}/leituras/{leitura}', [BarragemController::class, 'destroyLeitura'])->name('barragens.leituras.destroy');
    Route::put('/barragens/{barragem}/leituras/{leitura}/validate', [BarragemController::class, 'validateLeitura'])->name('barragens.validateLeitura');
    Route::get('/barragens/{barragem}/historico', [BarragemController::class, 'getHistorico'])->name('barragens.historico');
    Route::get('/api/barragens/{barragem}/chart-data', [App\Http\Controllers\Api\ChartController::class, 'barragemData'])->name('api.barragens.chart-data');

    // Curvas Cota-Volume
    Route::get('/barragens/{barragem}/curvas-volume', [App\Http\Controllers\CurvaVolumeController::class, 'index'])->name('barragens.curvas-volume.index');
    Route::post('/barragens/{barragem}/curvas-volume', [App\Http\Controllers\CurvaVolumeController::class, 'store'])->name('barragens.curvas-volume.store');
    Route::put('/barragens/{barragem}/curvas-volume/{curva}', [App\Http\Controllers\CurvaVolumeController::class, 'update'])->name('barragens.curvas-volume.update');
    Route::delete('/barragens/{barragem}/curvas-volume/{curva}', [App\Http\Controllers\CurvaVolumeController::class, 'destroy'])->name('barragens.curvas-volume.destroy');
    Route::get('/api/barragens/{barragem}/curvas-volume/chart', [App\Http\Controllers\CurvaVolumeController::class, 'getChartData'])->name('api.barragens.curvas-volume.chart');
    
    // Estações
    Route::resource('estacoes', EstacaoController::class)->parameters([
        'estacoes' => 'estacao'
    ]);

    // Rotas para leituras de estações
    Route::get('/estacoes/{estacao}/leituras', [EstacaoController::class, 'leituras'])->name('estacoes.leituras');
    Route::get('/estacoes/{estacao}/leituras/create', [EstacaoController::class, 'createLeitura'])->name('estacoes.leituras.create');
    Route::get('/estacoes/{estacao}/leituras/import', [EstacaoController::class, 'showImportForm'])->name('estacoes.leituras.import');
    Route::get('/estacoes/{estacao}/leituras/{leitura}/edit', [EstacaoController::class, 'editLeitura'])->name('estacoes.leituras.edit');
    Route::post('/estacoes/{estacao}/leituras', [EstacaoController::class, 'storeLeitura'])->name('estacoes.storeLeitura');
    Route::put('/estacoes/{estacao}/leituras/{leitura}', [EstacaoController::class, 'updateLeitura'])->name('estacoes.leituras.update');
    Route::delete('/estacoes/{estacao}/leituras/{leitura}', [EstacaoController::class, 'destroyLeitura'])->name('estacoes.leituras.destroy');

    // Gestão de fórmulas de caudal
    Route::get('/estacoes/{estacao}/formulas-caudal', [App\Http\Controllers\FormulaCaudalController::class, 'index'])->name('estacoes.formulas-caudal.index');
    Route::post('/estacoes/{estacao}/formulas-caudal', [App\Http\Controllers\FormulaCaudalController::class, 'store'])->name('estacoes.formulas-caudal.store');
    Route::patch('/estacoes/{estacao}/formulas-caudal/{formula}/toggle', [App\Http\Controllers\FormulaCaudalController::class, 'toggle'])->name('estacoes.formulas-caudal.toggle');
    Route::delete('/estacoes/{estacao}/formulas-caudal/{formula}', [App\Http\Controllers\FormulaCaudalController::class, 'destroy'])->name('estacoes.formulas-caudal.destroy');

    // APIs para funcionalidades avançadas de leituras
    Route::get('/api/estacoes/{estacao}/leituras/verificar/{data}', [EstacaoController::class, 'verificarLeitura'])->name('api.estacoes.verificar-leitura');
    Route::get('/api/estacoes/{estacao}/historico/{data}', [EstacaoController::class, 'getDadosHistoricos'])->name('api.estacoes.historico');
    Route::post('/api/estacoes/{estacao}/leituras/importar', [EstacaoController::class, 'importarLeituras'])->name('api.estacoes.importar-leituras');
    Route::post('/api/estacoes/{estacao}/leituras/preview', [EstacaoController::class, 'previewLeituras'])->name('api.estacoes.preview-leituras');
    Route::get('/estacoes/{estacao}/leituras/exportar', [EstacaoController::class, 'exportarLeituras'])->name('estacoes.exportar-leituras');

    // Templates de importação
    Route::get('/templates/leituras/{tipo}/{formato?}', [App\Http\Controllers\TemplateController::class, 'gerarTemplate'])->name('templates.leituras');
    Route::get('/api/templates/{tipo}', [App\Http\Controllers\TemplateController::class, 'listarTemplates'])->name('api.templates.listar');
    
    // Bacias Hidrográficas
    Route::resource('bacias', BaciaHidrograficaController::class)->parameters([
        'bacias' => 'bacia'
    ]);
    Route::get('/api/bacias', [BaciaHidrograficaController::class, 'api'])->name('api.bacias');
    
    // Alertas
    Route::resource('alertas', AlertaController::class);
    Route::post('/alertas/{alerta}/resolve', [AlertaController::class, 'resolve'])->name('alertas.resolve');
    Route::post('/alertas/{alerta}/cancel', [AlertaController::class, 'cancel'])->name('alertas.cancel');
    Route::get('/alertas/ativos/json', [AlertaController::class, 'getAtivos'])->name('alertas.ativos');
    
    // Notificações
    Route::prefix('notifications')->group(function () {
        Route::get('/config', [NotificationConfigController::class, 'index'])->name('notifications.config');
        Route::put('/config', [NotificationConfigController::class, 'update'])->name('notifications.update');
        Route::post('/test-sms', [NotificationConfigController::class, 'testSms'])->name('notifications.test-sms');
        Route::get('/historico', [NotificationConfigController::class, 'historico'])->name('notifications.historico');
        Route::post('/reenviar/{notificacao}', [NotificationConfigController::class, 'reenviar'])->name('notifications.reenviar');
        
        // Admin routes
        // Route::middleware(['can:admin'])->group(function () { // Comentado temporariamente
        Route::group([], function () {
            Route::get('/dashboard', [NotificationConfigController::class, 'dashboard'])->name('notifications.dashboard');
            Route::get('/configuracoes', [NotificationConfigController::class, 'configuracoes'])->name('notifications.configuracoes');
            Route::post('/relatorio-diario', [NotificationConfigController::class, 'enviarRelatorioDiario'])->name('notifications.relatorio-diario');
            Route::get('/api/stats', [NotificationConfigController::class, 'apiStats'])->name('notifications.api.stats');
        });
    });
    
    // Configurações do Sistema
    Route::prefix('configuracoes')->group(function () {
        Route::get('/', [ConfiguracoesController::class, 'index'])->name('configuracoes.index');
        
        // SMS/Twilio
        Route::get('/sms', [ConfiguracoesController::class, 'sms'])->name('configuracoes.sms');
        Route::put('/sms', [ConfiguracoesController::class, 'updateSms'])->name('configuracoes.sms.update');
        Route::post('/sms/test', [ConfiguracoesController::class, 'testSms'])->name('configuracoes.sms.test');
        
        // Email
        Route::get('/email', [ConfiguracoesController::class, 'email'])->name('configuracoes.email');
        Route::put('/email', [ConfiguracoesController::class, 'updateEmail'])->name('configuracoes.email.update');
        
        // Alertas
        Route::get('/alertas', [ConfiguracoesController::class, 'alertas'])->name('configuracoes.alertas');
        Route::put('/alertas', [ConfiguracoesController::class, 'updateAlertas'])->name('configuracoes.alertas.update');
        
        // Sistema
        Route::get('/sistema', [ConfiguracoesController::class, 'sistema'])->name('configuracoes.sistema');
        Route::put('/sistema', [ConfiguracoesController::class, 'updateSistema'])->name('configuracoes.sistema.update');
        Route::post('/sistema/clear-caches', [ConfiguracoesController::class, 'clearCaches'])->name('configuracoes.sistema.clear-caches');
        
        // Informações
        Route::get('/info', [ConfiguracoesController::class, 'info'])->name('configuracoes.info');
    });
    
    // Boletins
    Route::resource('boletins', BoletimController::class)->parameters([
        'boletins' => 'boletim'
    ]);
    Route::post('/boletins/{boletim}/approve', [BoletimController::class, 'approve'])->name('boletins.approve');
    Route::post('/boletins/{boletim}/publish', [BoletimController::class, 'publish'])->name('boletins.publish');
    Route::get('/boletins/{boletim}/download-pdf', [BoletimController::class, 'downloadPdf'])->name('boletins.download-pdf');
    Route::get('/boletins/{boletim}/export/{format}', [BoletimController::class, 'export'])->name('boletins.export');
    
    // Importação
    Route::get('/importacao', [ImportacaoController::class, 'index'])->name('importacao.index');
    Route::post('/importacao/barragens', [ImportacaoController::class, 'importBarragens'])->name('importacao.barragens');
    Route::post('/importacao/barragens-completo', [ImportacaoController::class, 'importBarragensCompleto'])->name('importacao.barragens-completo');
    Route::post('/importacao/estacoes', [ImportacaoController::class, 'importEstacoes'])->name('importacao.estacoes');
    Route::post('/importacao/leituras', [ImportacaoController::class, 'importLeituras'])->name('importacao.leituras');
    
    // Gráficos e Relatórios
    Route::prefix('graficos')->name('charts.')->group(function () {
        Route::get('/', [ChartsController::class, 'index'])->name('index');
        Route::get('/barragens', [ChartsController::class, 'barragens'])->name('barragens');
        Route::get('/barragens/{barragem}', [ChartsController::class, 'barragemDetail'])->name('barragem-detail');
        Route::get('/estacoes', [ChartsController::class, 'estacoes'])->name('estacoes');
        Route::get('/estacoes/{estacao}', [ChartsController::class, 'estacaoDetail'])->name('estacao-detail');
        Route::get('/regional', [ChartsController::class, 'regional'])->name('regional');
        
        // API endpoints for charts data
        Route::get('/api/barragem/{barragem}/data', [ChartsController::class, 'apiBarragemData'])->name('api.barragem-data');
        Route::get('/api/estacao/{estacao}/data', [ChartsController::class, 'apiEstacaoData'])->name('api.estacao-data');
    });

    // Configurações da Organização
    Route::get('/organizacao-settings', [SystemSettingsController::class, 'index'])->name('settings.index');
    Route::put('/organizacao-settings', [SystemSettingsController::class, 'update'])->name('settings.update');

    // Mapas Interativos
    Route::prefix('mapas')->name('mapas.')->group(function () {
        Route::get('/', [MapController::class, 'show'])->name('show');
        Route::get('/dados', [MapController::class, 'dados'])->name('dados');
        Route::get('/shapes/{shape}', [MapController::class, 'getShape'])->name('shapes');
        Route::get('/shapefiles/{shape}/{type}', [MapController::class, 'getShapefile'])->name('shapefiles');
    });

    // Gráficos de Barragem (similar ao Excel) - Usando apenas dados reais
    Route::get('/graficos-barragem', function () {
        $barragens = \App\Models\Barragem::with('ultima_leitura')->get();
        $resumo = [
            'total_barragens' => $barragens->count(),
            'media_enchimento' => $barragens->filter(fn($b) => $b->ultima_leitura)
                                            ->avg(fn($b) => $b->ultima_leitura->percentagem_enchimento ?? 0)
        ];
        return view('graficos-barragem', compact('barragens', 'resumo'));
    })->name('graficos-barragem');
});


require __DIR__.'/auth.php';
